#ifndef LOGFILE_H
#define LOGFILE_H

#include "Message.h"
#include <string>
#include <vector>
#include <map>

namespace DataLog
{

    class RawMessage;

	class LogFile
	{
		typedef Message* (*FnPtrType)(FILE*);
	public:
		LogFile();
		~LogFile();
		// Opens a Logfile. Returns false if an error occured (wrong header, file does not exist, ...)
		bool open(const std::string& fileName);
		// Seeks to a certain position (specified in seconds starting from 0) within the logfile. Returns false if an error occured.
		// Keep in mind that timestamps do not need to start at 0 - so basically seek is relative to the first timestamp
		bool seek(int toSec);

		// Registers a message that is supposed to be read from the LogFile. Example: myLog->handleMessage<MessageIWantToRead>();
		// MessageIWantToRead needs to fulfill certain conventions such as having a MSG_NAME static member
		template<typename T>
		void handleMessage()
		{
			m_msgHandlers[detail::simpleHash(T::MSG_NAME)]=&T::readMessage;
		}

		// Attempts to read a message from the log. If it is unsuccessful (eof reached for example), 0 is returned. Otherwise
		// a pointer to a Message is returned. The actual message read can be determined using dynamic_cast for example:
		// Message* msg=myLog->readNextMessage();
		// MyConcreteMessage* mcm=dynamic_cast<MyConcreteMessage*>(msg);
		// if (mcm) { ... /* do something with test */ ... }
		//
		// If you prefer not to use RTTI, you can alternatively check the name of the message:
		// if (msg->getMessageName()=="MyConcreteMessage")
		// {
		//    MyConcreteMessage* mcm2=(MyConcreteMessage*)msg;
		//    //do something with mcm2
		// }
		Message* readNextMessage(bool returnRawMessages=false);
        
        RawMessage * readRawMessage();

        bool create(const std::string& fileName, unsigned int index_size=1000);
        bool writeMessage(Message * p_message);
        int getIndexSize() const { return m_replayIndex.size();}
	private:
		FILE* m_r; // for reading
        FILE* m_w; // for writing
		long long m_fileSize;
		
		int m_replayLength; // in seconds
		std::vector<long long> m_replayIndex; // maps seconds to positions in replay file
        int m_currIndexSec;
        
		std::map<unsigned int, FnPtrType> m_msgHandlers;

		// no copying
		void operator=(const LogFile&);
		LogFile(const LogFile&);
	};
	
}

#endif
