#ifndef IOUTILS_H_
#define IOUTILS_H_

//#define USE_MRPT

#include <nrt/Core/Util/CompilerUtil.H>

#ifdef USE_MRPT
NRT_BEGIN_UNCHECKED_INCLUDES;
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
#include <mrpt/math.h>
#include <mrpt/utils/CFileGZInputStream.h>
#pragma GCC diagnostic pop
NRT_END_UNCHECKED_INCLUDES;
#endif // USE_MRPT

#include <nrt/Core/Design/Optional.H>
#include <nrt/PointCloud2/PointCloud2.H>
#include <nrt/Core/Typing/Enum.H>
#include <memory>
#include <fstream>

#include "MessageReader.H"

NRT_DEFINE_ENUM_CLASS(DataSource, (nrtlog) 
#ifdef USE_MRPT
    (mrptlog)
#endif // USE_MRPT
    (karlsruhe) (koblenz));

// stuff copied from the velodyne driver
namespace velodyne_koblenz
{
  //! Data from a single laser
  struct LaserData
  {
      nrt::uint16 distance; // 2mm increments, 0 = no return within 100m
      nrt::byte intensity;  // 0-255, 255 = most intense return
  } __attribute__((packed));

  //! Enum for the "select" value used to descriminate between firing blocks
  enum LaserBlockSelect { BLOCK_0_TO_31  = 0xEEFF, BLOCK_32_TO_63 = 0xDDFF };

  //! Block of lasers
  struct LaserBlock
  {
      nrt::uint16 id;       // should always be 0xEEFF (first block, HDL-32E) or 0xDDFF (second block, HDL-64)
      nrt::uint16 rotation; // 0-35999 in degrees * 100
      LaserData   laserdata[32];
  } __attribute__((packed));

  //! A raw data packet from the lasers
  struct LaserPacket
  {
      LaserBlock laserblock[12];
      nrt::uint32 timestamp; // usec since the top of the hour, bytes in reverse order
      nrt::byte unused1;
      nrt::byte unused2;
  } __attribute__((packed));

  //! Extra laser data that may be saved in the cloud in addition to intensity
  struct ExtraLaserData
  {
      nrt::uint16 rotation; // 0-35999 in degrees * 100
      nrt::byte lasernum;   // Laser sequence number [0..31] (see Velodyne doc and also LaserAngle in VelodyneLaser.C)

      template <class Archive> inline
      void serialize(Archive & ar)
      { ar( rotation, lasernum ); }
  };
}

class KoblenzReader
{
  protected:
    //! General Koblenz file header
    struct Header
    {
      uint8_t magic[4];
      uint16_t major_version;
      uint16_t minor_version;
    } __attribute__((packed));

    //! Index information
    /*! One index entry exists per second of data recorded, with each index potentially having many messages */
    struct IndexHeader
    {
      uint32_t size; // number of indices
      /* uint64_t index[size] */ // location of message index (absolute)
    } __attribute__((packed));

    //! Message Types
    enum MessageType : int32_t
    {
      OBDDataM = 0x00014043,
      GPSDataM = 0x00014A32,
      ImageM = 0x000109C9,
      RobotPoseM = 0x0001E342,
      VelodyneRawDataM = 0x0003112B
    };

    struct MessageHeader
    {
      uint32_t size; // size in bytes including header
      uint8_t valid; // 1 for valid
      MessageType type; // see above enum
      int32_t version; // always 100 for now
      double timestamp; // time in ms since startup
      /* char data[size - 17] */ // the actual data
    } __attribute__((packed));

    struct OBDDataMessage
    {
      int32_t speed; // in km/h
      int32_t rpm; // engine rpm
      float unused0;
      int32_t unused1;
      int32_t unused2;
      float throttle_position;
      int32_t unused3;
    } __attribute__((packed));

    struct GPSDataMessage
    {
      int32_t time_hour; // UTC time
      int32_t time_minute;
      int32_t time_second;
      int32_t warning; // 0 or 1
      double latitude;
      double longitude;
      float speed; // km/h
      float course;
      int32_t date_day;
      int32_t date_month;
      int32_t date_year;
      int32_t quality; // 0 = invalid, 1 = gps, 2 = dgps, 6 = estimated
      int32_t number_sats;
      float accuracy_hdop;
      float height;
      float geoid_height;
      float accuracy_vdop;
      float accuracy_pdop;
    } __attribute__((packed));

    struct ImageMessage
    {
      int32_t id;
      int32_t is_compressed; // always true, using JPEG
      int32_t width;
      int32_t height;
      uint32_t size;
      /* uint8_t data[size] */ // image data
    } __attribute__((packed));

    struct RobotPoseMessage
    {
      float orientation[4]; // quaternion
      float acceleration[3]; // accel vector
    } __attribute__((packed));

    struct VelodyneRawDataMessage
    {
      uint32_t num_packets;
      /* velodyne::LaserPacket packets[num_packets] */
    } __attribute__((packed));

  public:
    //! Does what you think it does
    KoblenzReader( std::string const & file );

    //! Get the next cloud, optionally ignoring up to skipFrames
    nrt::Optional<nrt::PointCloud2> getCloud( int skipFrames = 0 );

  protected:
    void addLaserBlock( velodyne_koblenz::LaserBlock const & lb, nrt::PointCloud2 & cloud,
        float const mindist, float const maxdist, bool const extra );

    bool addLaserPacket( velodyne_koblenz::LaserPacket const & packet, float const mindist = 1.0f,
        float const maxdist = 130.0f,
        bool const extra = false );


  private:
    nrt::PointCloud2 itsCurrCloud, itsNextCloud;
    uint16_t itsLastRotation;

    float cosVertAngle[64], sinVertAngle[64], cosRotAngle[36000], sinRotAngle[36000];

    std::ifstream itsFile;
};

//! A class for loading velodyne data from various sources
class VelodyneIO
{
  public:
    VelodyneIO( std::string const & file, DataSource source );

    //! Gets the next frame
    nrt::Optional<nrt::PointCloud2> getCloud();

    //! Gets a specific frame, assuming it exceeds the last frame
    nrt::Optional<nrt::PointCloud2> getCloud( int frame );

  protected:
    nrt::Optional<nrt::PointCloud2> getCloudNRTLog( int frame );
    nrt::Optional<nrt::PointCloud2> getCloudMRPTRawLog( int frame );
    nrt::Optional<nrt::PointCloud2> getCloudKarlsruhe( int frame );
    nrt::Optional<nrt::PointCloud2> getCloudKoblenz( int frame );

  private:
    DataSource itsDataSource;
    std::unique_ptr<MessageReader> itsReader;
    std::unique_ptr<std::ifstream> itsFileReader;
    std::unique_ptr<KoblenzReader> itsKoblenz;
#ifdef USE_MRPT
    std::unique_ptr<mrpt::utils::CFileGZInputStream> itsRawLog;
#endif // USE_MRPT
    int itsLastFrame;
    bool itsFinished;
};

#endif // IOUTILS_H_
