#include "PlaneSLAM.H"
#include <nrt/Core/Model/Manager.H>
#include <nrt/Graphics/ShapeRendererBasic.H>
#include <nrt/Graphics/Shapes.H>
#include <nrt/ImageProc/Drawing/ColorMapping.H>
#include <nrt/PointCloud2/Common/Transforms.H>
#include <DetectedPlane.H>
#include <PlaneAssociation.H>
#include <details/RefinePlane.H>
#include <details/Polygonalization.H>
#include "DrawingUtils.H"

using namespace nrt;

struct Wall
{
  Wall() = default;
  Wall(Eigen::Hyperplane<float, 3> p_, float w_, float h_) :
    plane(p_), width(w_), height(h_) {}

  Eigen::Hyperplane<float, 3> plane;
  float width, height;
};

// ######################################################################
nrt::Indices appendWallPoints(Wall const & wall, nrt::PointCloud2 & cloud)
{
  std::random_device rd;
  std::default_random_engine e(rd());
  std::uniform_real_distribution<float> x_dist(-wall.width/2, wall.width/2);
  std::normal_distribution<float> y_dist(0, .0);
  std::uniform_real_distribution<float> z_dist(-wall.height/2, wall.height/2);

  Eigen::Quaternionf q;
  q.setFromTwoVectors(Eigen::Vector3f(0, 1, 0), wall.plane.normal());
  Eigen::Affine3f transform = Eigen::Affine3f::Identity();
  transform.rotate(q);
  transform.translate(Eigen::Vector3f(0, -wall.plane.offset(), 0));

  float const points_per_sqmeter = 100;
  int const num_points = wall.width * wall.height * points_per_sqmeter;

  nrt::Indices indices;
  size_t end_index = cloud.size() + num_points;
  for(size_t index=cloud.size(); index<end_index; ++index)
  {
    auto p = transform * Eigen::Vector3f(x_dist(e), y_dist(e), z_dist(e));
    cloud.insert<Distance>({PointCloud2::Geometry{p.x(), p.y(), p.z()}, p.norm()});
    indices.push_back(index);
  }

  return indices;
}

// ######################################################################
std::vector<DetectedPlane> makeObservations(std::vector<Wall> const & world, Eigen::Isometry3d const & pose)
{

  nrt::PointCloud2 worldCloud;
  worldCloud.addField<Distance>();
  std::vector<DetectedPlane> observations;

  // Create the world points, and get the indices for the planes
  for(Wall const & wall : world)
  {
    auto plane = wall.plane;

    DetectedPlane observation;
    observation.indices = appendWallPoints(Wall(plane, wall.width, wall.height), worldCloud);
    observation.area = wall.width * wall.height;
    observations.push_back(observation);
  }

  Eigen::Isometry3f pose_inv = pose.cast<float>().inverse();

  // Transform the world cloud
  for(auto p : worldCloud)
  {
    Eigen::Vector3f p_t = pose_inv * p.geometry().getVector3Map();
    p.geometry() = PointCloud2::Geometry(p_t.x(), p_t.y(), p_t.z());
  }

  // Fit planes and convex hulls to the cloud
  for(auto & observation : observations)
  {
    observation = refinePlaneLeastSquares(observation, worldCloud);
    observation.hull = convexHullPolygonalization(observation, worldCloud);
  }

  return observations;
}

// ######################################################################
int main(int argc, const char ** argv)
{
  Manager mgr(argc, argv);
  auto renderer = mgr.addComponent<graphics::ShapeRendererBasic>("Renderer");
  mgr.launch();

  std::vector<Wall> world = {
    //|     Plane     |  w  |  h  |
    {{{ 0,  0,  1},  1},   2,   20}, // floor
    {{{ 0,  0, -1},  1},   2,   20}, // ceiling
    {{{ 1,  0,  0},  1},  20,    2}, // left wall
    {{{-1,  0,  0},  1},  20,    2}, // right wall
    {{{ 0,  1,  0}, 10},   2,    2}, // behind wall
    {{{ 0, -1,  0}, 10},   2,    2}, // forward wall
  };


  bool next_frame = false;
  renderer->setKeyboardCallback([&next_frame](std::vector<graphics::ShapeRenderer::KeyboardPress> const & keys)
      {
        for(auto key : keys)
          if(key.release == false)
            next_frame = true;
      });

  std::vector<Eigen::Isometry3d> real_poses = { Eigen::Isometry3d::Identity() };
  std::vector<Eigen::Isometry3d> odom_poses = { Eigen::Isometry3d::Identity() };

  PlaneSLAM slam;

  std::vector<DetectedPlane> last_observations = makeObservations(world, real_poses.back());
  int t = 0;
  while(true)
  {
    std::cout << "-------------- " << t << " --------------" << std::endl;
    // Update the real world
    Eigen::Isometry3d real_odometry = Eigen::Isometry3d::Identity();
    if(t > 3)
    {
      real_odometry.rotate(Eigen::AngleAxisd(M_PI/40, Eigen::Vector3d::UnitZ()));
      real_odometry.translate(Eigen::Vector3d(0, .01, 0));
    }

    real_poses.push_back(real_odometry * real_poses.back());

    std::cout << "Real Pose: " << real_poses.back().translation().transpose() << std::endl;

    nrt::PointCloud2 worldPoints;
    worldPoints.addField<Distance>();
    for(Wall wall : world)
      appendWallPoints(wall, worldPoints);

    // Observe the world
    std::vector<DetectedPlane> observations = makeObservations(world, real_poses.back());

    // Find the correspondences and transform
    nrt::Correspondences correspondences;

    // MUMC transform is from observations (current) to previous frame
    //    The inverse will take us from previous to current (N-1 to N)
    auto calc_transform = findTransformMUMCFast(last_observations, observations, correspondences).inverse();

    // odometry poses consists of transforms that take us from the origin to their location
    odom_poses.push_back(calc_transform * odom_poses.back());

    // Update SLAM
    // slam is passed a transformation from the last frame to the current frame (N-1 to N)
    auto slam_poses = slam.update(calc_transform, correspondences, last_observations, observations);
    auto map = slam.getRenderMap();

    //std::cout << "Estimated Pose: " << slam_poses.back().translation().transpose() << std::endl;
    //for(auto p : map)
    //  std::cout << std::string(std::get<0>(p)) << " = " << std::get<1>(p).coeffs().transpose() << " :: " << std::get<2>(p).size() << " points in hull" << std::endl;

    while(!next_frame)
    {
      renderer->initFrame();

      //drawPlanes(renderer, observations, real_poses.back());
      //for(auto o : observations)
      //  createPlanePolygon(o.plane.transform(Eigen::Affine3f(real_poses.back().cast<float>())),
      //      getNormalColor(o.plane.normal(), 128), 2).render(*renderer);

      // Draw the wall points
      graphics::PointCloud worldCloudGraphics(worldPoints, 1.0);
      worldCloudGraphics.setDisplayType(graphics::PointCloud::DisplayType::Cube);
      worldCloudGraphics.setPointSize(0.01);

      worldCloudGraphics.render(*renderer);

      // Draw the origin axes
      graphics::Axes(slam_poses.back().cast<float>()).render(*renderer);

      Eigen::Affine3f axesOdom(odom_poses.back().cast<float>());
      axesOdom.scale(.75);
      graphics::Axes(axesOdom).render(*renderer);

      Eigen::Affine3f axesReal(real_poses.back().cast<float>());
      axesReal.scale(.5);
      graphics::Axes(axesReal).render(*renderer);

      // Draw the real poses
      for(float i=0; i<float(real_poses.size())-1; ++i)
        nrt::graphics::Sphere(
            Eigen::Affine3f(real_poses[i].cast<float>()).scale(0.05),
            nrt::forev(),
            PixRGB<byte>(255)).render(*renderer);

      // Draw the odometry poses
      for(float i=0; i<float(odom_poses.size())-1; ++i)
        nrt::graphics::Sphere(
            Eigen::Affine3f(odom_poses[i].cast<float>()).scale(0.05),
            nrt::forev(),
            nrt::jetPixel(i, 0.f, float(odom_poses.size()))).render(*renderer);

      // Draw the slam poses
      for(float i=0; i<float(slam_poses.size())-1; ++i)
        nrt::graphics::Box(
            Eigen::Affine3f(slam_poses[i].cast<float>()).scale(0.1),
            nrt::forev(),
            nrt::jetPixel(i, 0.f, float(slam_poses.size()))).render(*renderer);

      drawSLAMMap(renderer, map);

      for(auto p : map)
        createPlanePolygon(std::get<1>(p), getNormalColor(std::get<1>(p).normal(), 128), 2).render(*renderer);

      renderer->renderFrame();
    }
    next_frame = false;

    last_observations = observations;
    ++t;
  }



  return 0;
}
