#include "LiDARPlaneDetectorDetails/RefinePlane.H"
#include <valgrind/callgrind.h>

int main()
{
  CALLGRIND_STOP_INSTRUMENTATION;
  std::uniform_real_distribution<float> realrand(-5, 5);

  std::random_device rd;
  std::mt19937 gen(rd());

  Eigen::Hyperplane<float, 3> plane(Eigen::Vector3f(realrand(gen), realrand(gen), realrand(gen)).normalized(), realrand(gen));


  std::normal_distribution<> d(0,1);

  DetectedPlane detectedPlane;
  nrt::PointCloud2 cloud;
  for(size_t i=0; i<20000; ++i)
  {
    Eigen::Vector3f p = plane.projection(Eigen::Vector3f(realrand(gen), realrand(gen), realrand(gen)));
    p += Eigen::Vector3f(d(gen), d(gen), d(gen));
    cloud.insert({p.x(), p.y(), p.z()});
    detectedPlane.indices.push_back(i);
  }

  int tries = 300;

  std::array<std::vector<LiDARGroup>, 32> groups;

  auto slowStart = std::chrono::steady_clock::now();
  for(int i=0; i<tries; ++i)
    detectedPlane = refinePlaneFromPoints<32>(detectedPlane, groups, cloud, RefineFrom::Indices);
  auto slowStop = std::chrono::steady_clock::now();

  std::cout << "Plane: n[" << plane.normal().transpose() << "] rho: " << plane.offset() << std::endl;

  std::cout << "Plane: n[" << detectedPlane.plane.normal().transpose() << "] rho: " << detectedPlane.plane.offset() << std::endl;

  auto fastStart = std::chrono::steady_clock::now();

  CALLGRIND_START_INSTRUMENTATION;
  for(int i=0; i<tries; ++i)
    detectedPlane = refinePlaneFromPointsFast<32>(detectedPlane, groups, cloud, RefineFrom::Indices);
  CALLGRIND_STOP_INSTRUMENTATION;
  auto fastStop = std::chrono::steady_clock::now();

  std::cout << "Plane: n[" << detectedPlane.plane.normal().transpose() << "] rho: " << detectedPlane.plane.offset() << std::endl;

  std::cout << "Timing:" << std::endl;
  std::cout << "  Slow: " << std::chrono::duration_cast<std::chrono::microseconds>(slowStop - slowStart).count() / 1000.0 / float(tries) << "ms" << std::endl;
  std::cout << "  Fast: " << std::chrono::duration_cast<std::chrono::microseconds>(fastStop - fastStart).count() / 1000.0 / float(tries) << "ms" << std::endl;

  return 0;
}
