#include "PoseToPlaneFactor.H"
#include <gtsam/geometry/Pose3.h>
#include <gtsam/geometry/Point3.h>
#include <gtsam/nonlinear/Symbol.h>
#include <gtsam/slam/PriorFactor.h>
#include <gtsam/slam/BetweenFactor.h>
#include <gtsam/nonlinear/NonlinearFactorGraph.h>
#include <gtsam/nonlinear/ISAM2.h>
#include <gtsam/nonlinear/Values.h>
#include <gtsam/nonlinear/NonlinearFactor.h>
#include <map>

int main(int argc, char ** argv)
{
  using namespace gtsam;

  Symbol p0('p', 0), p1('p', 1), p2('p', 2), p3('p', 3);
  Symbol m0('m', 0), m1('m', 1), m2('m', 2);

  ISAM2Params parameters;
  parameters.relinearizeThreshold = 0.01;
  parameters.relinearizeSkip = 1;
  ISAM2 isam(parameters);

  NonlinearFactorGraph graph;
  Values initialEstimate;
  Values currentEstimate;

  // Add the first pose (p0)
  Pose3 prior(Rot3::identity(), Point3(0,0,0));
  noiseModel::Diagonal::shared_ptr priorNoise = noiseModel::Diagonal::Sigmas((Vector(6) << 0.01, 0.01, 0.01, 0.3, 0.3, 0.3));
  graph.add(PriorFactor<Pose3>(p0, prior, priorNoise));

  // Add the first plane measurement
  noiseModel::Diagonal::shared_ptr measurementNoise = noiseModel::Diagonal::Sigmas((Vector(4) << 0.2, 0.2, 0.2, 0.5));
  graph.add(PoseToPlaneFactor(PlaneValue(1, 0, 0, 5.0), measurementNoise, p0, m0));
  graph.add(PoseToPlaneFactor(PlaneValue(-1, 0, 0, 5.0), measurementNoise, p0, m1));

  // Add the initial estimates
  initialEstimate.insert(p0,   Pose3(Rot3::identity(),      Point3(0,  0, 0)));
  initialEstimate.insert(m0,   PlaneValue( 1, 0, 0, 5));
  initialEstimate.insert(m1,   PlaneValue(-1, 0, 0, 5));

  // Update iSAM
  isam.update(graph, initialEstimate);
  currentEstimate = isam.calculateBestEstimate();
  std::cout << "======================================" << std::endl;
  currentEstimate.print("Frame 0 Estimate:\n");

  // reset working graph and estimates
  graph.resize(0);
  initialEstimate.clear();

  // Add the odometry from p0 to p1
  noiseModel::Diagonal::shared_ptr odometryNoise = noiseModel::Diagonal::Sigmas((Vector(6) << 0.1, 0.1, 0.1, 0.3, 0.3, 0.3));
  Pose3 odometry1(Rot3::identity(), Point3(2, 3, 0));
  graph.add(BetweenFactor<Pose3>(p0, p1, odometry1, odometryNoise));

  graph.add(PoseToPlaneFactor(PlaneValue(1, 0, 0, 7),  measurementNoise, p1, m0));
  graph.add(PoseToPlaneFactor(PlaneValue(-1, 0, 0, 3), measurementNoise, p1, m1));

  initialEstimate.insert(p1,   Pose3(Rot3::identity(),      Point3(2,  3, 0)));

  // Update iSAM
  isam.update(graph, initialEstimate);
  currentEstimate = isam.calculateBestEstimate();
  std::cout << "======================================" << std::endl;
  currentEstimate.print("Frame 1 Estimate:\n");

  // reset working graph and estimates
  graph.resize(0);
  initialEstimate.clear();

  // Add the odometry from p1 to p2
  Pose3 odometry2(Rot3::Rz(-M_PI/4.), Point3(0, 3, 0));
  graph.add(BetweenFactor<Pose3>(p1, p2, odometry2, odometryNoise));

  graph.add(PoseToPlaneFactor(PlaneValue(7.0/std::sqrt(2.0), 7.0/std::sqrt(2.0), 0, 7), measurementNoise, p2, m0));
  graph.add(PoseToPlaneFactor(PlaneValue(-3.0/std::sqrt(2.0), -3.0/std::sqrt(2.0), 0, 3), measurementNoise, p2, m1));
  graph.add(PoseToPlaneFactor(PlaneValue(4.0/std::sqrt(2.0), -4.0/std::sqrt(2.0), 0, 4), measurementNoise, p2, m2));

  initialEstimate.insert(p2,   Pose3(Rot3::Rz(-M_PI/4.0),   Point3(2,  6, 0)));
  initialEstimate.insert(m2,   PlaneValue(0, -1, 0, 10));

  // Update iSAM
  isam.update(graph, initialEstimate);
  currentEstimate = isam.calculateBestEstimate();
  std::cout << "======================================" << std::endl;
  currentEstimate.print("Frame 2 Estimate:\n");

  // reset working graph and estimates
  graph.resize(0);
  initialEstimate.clear();

  // Add the odometry from p2 to p3
  //Pose3 odometry3(Rot3::Rz(M_PI/4.), Point3(-2, 0, 0));
  Pose3 odometry3(Rot3::identity(), Point3(-2, 0, 0));
  graph.add(BetweenFactor<Pose3>(p2, p3, odometry3, odometryNoise));

  //graph.add(PoseToPlaneFactor(PlaneValue(1, 0, 0, 5), measurementNoise, p3, m0));
  //graph.add(PoseToPlaneFactor(PlaneValue(0,-1, 0, 2), measurementNoise, p3, m2));

  initialEstimate.insert(p3,   Pose3(Rot3::identity(), Point3(0,  8, 0)));

  // Update iSAM
  isam.update(graph, initialEstimate);
  currentEstimate = isam.calculateBestEstimate();
  std::cout << "======================================" << std::endl;
  currentEstimate.print("Frame 3 Estimate:\n");

  return 0;
}
