#pragma once

#include <gtsam/nonlinear/NonlinearFactor.h>
#include <gtsam/geometry/Pose3.h>
#include <gtsam/nonlinear/Symbol.h>
#include <gtsam/geometry/Point3.h>
#include "PlaneValue.H"

class PlaneOverlapFactor : public gtsam::NoiseModelFactor2<gtsam::Pose3, PlaneValue>
{
  public:
    typedef gtsam::NoiseModelFactor2<gtsam::Pose3, PlaneValue> Base;

    PlaneOverlapFactor(PlaneOverlapFactor const & other) = default;

    PlaneOverlapFactor(PlaneValue const & measured, gtsam::SharedNoiseModel const & model,
        gtsam::Symbol poseKey, gtsam::Symbol planeKey) :
      Base(model, poseKey, planeKey),
      plane_measured_(measured),
      poseKey_(poseKey), planeKey_(planeKey),
      poseName_(std::string(poseKey)),
      planeName_(std::string(planeKey))
  { }

    gtsam::NonlinearFactor::shared_ptr clone() const override
    {
      return boost::static_pointer_cast<gtsam::NonlinearFactor>(gtsam::NonlinearFactor::shared_ptr(new PlaneOverlapFactor(*this)));
    }

    void print(std::string const & s = "", gtsam::KeyFormatter const & keyFormatter = gtsam::DefaultKeyFormatter) const override
    {
      std::cout << s << "PlaneOverlapFactor, z = "; plane_measured_.print();
      Base::print("", keyFormatter);
    }

    gtsam::Vector evaluateError(gtsam::Pose3 const & pose, PlaneValue const & plane_map,
        boost::optional<gtsam::Matrix&> H1 = boost::none, boost::optional<gtsam::Matrix&> H2 = boost::none) const override
    {
      // Input transform parameters T (R, t)
      gtsam::Matrix3 const & R = pose.rotation().matrix();
      gtsam::Matrix3 const Rt = R.transpose();
      gtsam::Vector const & t  = pose.translation().vector();

      // Input global (map) plane parameters p (n, d)
      gtsam::Vector const n_map = plane_map.n();
      double const d_map = plane_map.d();

      // Local (measured) plane parameters p (n, da
      gtsam::Vector const n_meas = plane_measured_.n();
      double const d_meas = plane_measured_.d();

      gtsam::Vector error(4,1);
      error.block<3,1>(0,0) = Rt*n_map - n_meas;
      error[3] = t.transpose()*n_map + d_map - d_meas;

      if(H1)
      {
        Eigen::Matrix<double, 4, 6> H;

        // dn / dw
        H.block<3,3>(0,0) = gtsam::skewSymmetric(Rt * n_map);

        // dd / dw
        H.block<1,3>(3,0) = Eigen::Matrix<double, 1, 3>::Zero();

        // dn / dt
        H.block<3,3>(0,3) = Eigen::Matrix<double, 3, 3>::Zero();

        // dd / dt
        //H.block<1,3>(3,3) = n_map.transpose();
        H.block<1,3>(3,3) = (Rt*n_map).transpose();

        *H1 = H;
      }

      if(H2)
      {
        Eigen::Matrix<double, 4, 4> H;

        // dn / dn
        H.block<3,3>(0, 0) = Rt;

        // dd / dn
        H.block<1,3>(3,0) = t.transpose();

        // dn / dd
        H.block<3,1>(0,3) = Eigen::Matrix<double, 3,1>::Zero();

        // dd / dd
        H(3,3) = 1;

        *H2 = H;
      }

      return error;
    }

    std::string poseName() { return poseName_; }
    std::string planeName() { return planeName_; }
    gtsam::Symbol poseKey() { return poseKey_; }
    gtsam::Symbol planeKey() { return planeKey_; }
    PlaneValue measurement() { return plane_measured_; }

  private:
    PlaneValue plane_measured_;
    gtsam::Symbol poseKey_, planeKey_;
    std::string poseName_, planeName_;
};


#if GTSAM_VERSION_MAJOR == 4
namespace gtsam
{
  template<> struct traits<PlaneOverlapFactor> : public Testable<PlaneOverlapFactor> {};
}
#endif
