/*! @file MessageLogging/MessageLoader/MessageLoaderModule.C */

#include "MessageLoaderModule.H"
#include <boost/archive/binary_iarchive.hpp>
#include <fstream>

// ######################################################################
template <class Port>
MessageLoaderSub<Port>::MessageLoaderSub(std::string const & instanceName) :
  nrt::Module(instanceName), itsQ(100)
{  }

// ######################################################################
template <class Port>
MessageLoaderSub<Port>::~MessageLoaderSub()
{  }

// ######################################################################
template <class Port>
void MessageLoaderSub<Port>::run()
{
  typedef typename Port::MsgType Msg;

  while (running())
  {
    if (itsQ.filled_size())
      try
      { 
        std::shared_ptr<MessageSaverData> data = itsQ.pop();

        if(nrt::CentralTimer::instance().get() > nrt::DurationSeconds(data->time)) continue;

        std::unique_ptr<Msg> msg(new Msg());
        std::istringstream is(data->sermsg); nrt::iarchive ir(is); ir(*msg);

        std::this_thread::sleep_until(nrt::CentralTimer::instance().epoch() + nrt::DurationSeconds(data->time));
        this->template post<Port>(msg); // will block until completion of all callbacks
      }
      catch (...) { nrt::warnAndIgnoreException(); } // can't throw here or will block our feeder
    else std::this_thread::sleep_for(std::chrono::milliseconds(10));
  }

  // Kill the contents of our queue so our feeded thread will unblock in case it was blocked pushing:
  while (itsQ.filled_size()) itsQ.pop();
}

// ######################################################################
template <class Port>
void MessageLoaderSub<Port>::dopost(std::shared_ptr<MessageSaverData> data)
{
  typedef typename Port::MsgType Msg;
  std::unique_ptr<Msg> msg(new Msg());
  std::istringstream is(data->sermsg); nrt::iarchive ir(is); ir(*msg);

  this->template post<Port>(msg);
}

// ######################################################################
template <class Port>
void MessageLoaderSub<Port>::postStop()
{
  // Flush our queue so we don't have old messages around at next start:
  while (itsQ.filled_size()) itsQ.pop();
}

// ######################################################################
// ######################################################################
// ######################################################################
MessageLoaderModule::MessageLoaderModule(std::string const & instanceName) :
    nrt::Module(instanceName),
    itsFilename(messageloadermodule::ParamDefFilename, this),
    itsLoop(messageloadermodule::ParamDefLoop, this),
    itsUseTrigger(messageloadermodule::ParamDefTrigger, this),
    itsUseOrderPost(messageloadermodule::ParamDefOrderPost, this),
    itsVelodyne(new MessageLoaderSub<messageloadersub::OutputVelodyne>(instanceName+"Velodyne")),
    itsXtion(new MessageLoaderSub<messageloadersub::OutputXtion>(instanceName+"Xtion")),
    itsXtionIm(new MessageLoaderSub<messageloadersub::OutputXtionIm>(instanceName+"XtionIm")),
    itsPose(new MessageLoaderSub<messageloadersub::OutputPose>(instanceName+"Pose")),
    itsIMU(new MessageLoaderSub<messageloadersub::OutputIMU>(instanceName+"IMU")),
    itsBBcloud(new MessageLoaderSub<messageloadersub::OutputBBcloud>(instanceName+"BBcloud")),
    itsBBdisparity(new MessageLoaderSub<messageloadersub::OutputBBdisparity>(instanceName+"BBdisparity")),
    itsBBleft(new MessageLoaderSub<messageloadersub::OutputBBleft>(instanceName+"BBleft")),
    itsBBright(new MessageLoaderSub<messageloadersub::OutputBBright>(instanceName+"BBright")),
    itsBBdepth(new MessageLoaderSub<messageloadersub::OutputBBdepth>(instanceName+"BBdepth")),
    itsMicrostrainIMU(new MessageLoaderSub<messageloadersub::OutputMicrostrainIMU>(instanceName+"MicrostrainIMU"))
{
  // Register our subs:
  addSubComponent(itsVelodyne);
  addSubComponent(itsXtion);
  addSubComponent(itsXtionIm);
  addSubComponent(itsPose);
  addSubComponent(itsIMU);
  addSubComponent(itsBBcloud);
  addSubComponent(itsBBdisparity);
  addSubComponent(itsBBleft);
  addSubComponent(itsBBright);
  addSubComponent(itsBBdepth);
  addSubComponent(itsMicrostrainIMU);
}

// ######################################################################
MessageLoaderModule::~MessageLoaderModule()
{  }

// ######################################################################
void MessageLoaderModule::run()
{
  do
  {
    // Open input file and archive:
    NRT_INFO("Loading from file: " << itsFilename.getVal());
    std::ifstream ifs(itsFilename.getVal());
    if (ifs.is_open() == false)
      throw nrt::exception::ModuleException(this, "Failed to open input file", itsFilename.getVal());
    nrt::iarchive ar(ifs);

    // Main loop:
    size_t count = 0;
    while (running() && ! ifs.eof())
    {
      // Possibly wait for next trigger; check will return its results that contain futures that will block us:
      if (itsUseTrigger.getVal()) check<messageloadermodule::InputTrigger>(nrt::MessageCheckerPolicy::UnseenBlock);

      // Read one message from the archive:
      std::shared_ptr<MessageSaverData> data(new MessageSaverData);
      ar (*data);

      //NRT_INFO("Loaded message of type " << data->key);

      if (itsUseOrderPost.getVal())
      {
        //Post the messages right away
        if (data->key ==      "messagesavermodule::InputVelodyne") itsVelodyne->dopost(data);
        else if (data->key == "messagesavermodule::InputXtion") itsXtion->dopost(data);
        else if (data->key == "messagesavermodule::InputXtionIm") itsXtionIm->dopost(data);
        else if (data->key == "messagesavermodule::InputPose") itsPose->dopost(data);
        else if (data->key == "messagesavermodule::InputIMU") itsIMU->dopost(data);
        else if (data->key == "messagesavermodule::InputBBcloud") itsBBcloud->dopost(data);
        else if (data->key == "messagesavermodule::InputBBdisparity") itsBBdisparity->dopost(data);
        else if (data->key == "messagesavermodule::InputBBleft") itsBBleft->dopost(data);
        else if (data->key == "messagesavermodule::InputBBright") itsBBright->dopost(data);
        else if (data->key == "messagesavermodule::InputBBdepth") itsBBdepth->dopost(data);
        else if (data->key == "messagesavermodule::InputMicrostrainIMU") itsMicrostrainIMU->dopost(data);
        else NRT_WARNING("Oooops, unknown message [" << data->key << "] -- IGNORED");
      } else {
        //Place the messages in a queue and post when its time is due
        if (data->key ==      "messagesavermodule::InputVelodyne") itsVelodyne->itsQ.push(data);
        else if (data->key == "messagesavermodule::InputXtion") itsXtion->itsQ.push(data);
        else if (data->key == "messagesavermodule::InputXtionIm") itsXtionIm->itsQ.push(data);
        else if (data->key == "messagesavermodule::InputPose") itsPose->itsQ.push(data);
        else if (data->key == "messagesavermodule::InputIMU") itsIMU->itsQ.push(data);
        else if (data->key == "messagesavermodule::InputBBcloud") itsBBcloud->itsQ.push(data);
        else if (data->key == "messagesavermodule::InputBBdisparity") itsBBdisparity->itsQ.push(data);
        else if (data->key == "messagesavermodule::InputBBleft") itsBBleft->itsQ.push(data);
        else if (data->key == "messagesavermodule::InputBBright") itsBBright->itsQ.push(data);
        else if (data->key == "messagesavermodule::InputBBdepth") itsBBdepth->itsQ.push(data);
        else if (data->key == "messagesavermodule::InputMicrostrainIMU") itsMicrostrainIMU->itsQ.push(data);
        else NRT_WARNING("Oooops, unknown message [" << data->key << "] -- IGNORED");
      }

      ++count;
      if (count % 100 == 0) NRT_INFO("Data Time " << data->time << " Loaded " << count << " messages so far...");

      // eof will not be set until past the end, hence the peek() here, in case we are right at the end:
      ifs.peek();
    }

    ifs.close();

  } while (running() && itsLoop.getVal());

  NRT_INFO("Done.");
}

// ######################################################################
// Don't forget this to be able to use your module as a runtime-loadable shared object
NRT_REGISTER_MODULE(MessageLoaderModule);
