
#include "LiDARPlaneDetector.H"
#include "details/FrameTiming.H"
#include "details/VoteForGroup.H"
#include "details/FindGroupsInRow.H"
#include "details/FilterMinLasers.H"
#include "details/SplitPlanes.H"
#include "details/RefinePlane.H"
#include "details/SmartMerge.H"
#include "details/RegionGrowing.H"
#include "details/Polygonalization.H"
#include "details/PolygonMerge.H"
#include "Filtering.H"
#include "PlaneDetectionCommon.H"
#include <functional>
#include <valgrind/callgrind.h>
#include <boost/format.hpp>

#include <Config/Config.H>

std::string planedetect::getEigenVersion()
{
  return std::to_string(EIGEN_WORLD_VERSION) + "."
       + std::to_string(EIGEN_MAJOR_VERSION) + "."
       + std::to_string(EIGEN_MINOR_VERSION);
}

using namespace std::placeholders;

// ######################################################################
bool operator<(LiDARGroup const & g1, LiDARGroup const & g2)
{ return g1.id < g2.id; }

// ######################################################################
template<size_t numRows>
LiDARPlaneDetector<numRows>::LiDARPlaneDetector(
    int const numTheta, int const numPhi, int const numRho, float const rhoMax,
    float const rhoSigma, bool voteSpread) :
  itsAccumulator(numTheta, numPhi, numRho, rhoMax, rhoSigma),
  itsVoteSpread(voteSpread),
  itsRowRange(numRows)
{
  for(size_t i=0; i<numRows; ++i) itsRowRange[i] = i;
}

// ######################################################################
template<size_t numRows>
void LiDARPlaneDetector<numRows>::setAccumulatorThreshold(float const threshold)
{
  itsAccumulator.setThreshold(threshold);
}

// ######################################################################
template<size_t numRows>
std::array<std::vector<LiDARGroup>, numRows> const & LiDARPlaneDetector<numRows>::getGroups()
{
  return itsGroups;
}

// ######################################################################
namespace
{
  void report(std::map<std::string, size_t> const & numPlanes)
  {
    std::cout << "---------------------" << std::endl;
    for(std::pair<std::string, size_t> n : numPlanes)
    {
      std::cout << boost::format("[%|-30|] : %|10t|%d") % n.first % n.second;
      std::cout << std::endl;
    }
    std::cout << "---------------------" << std::endl;
  }
}

// ######################################################################
template<size_t numRows>
std::vector<DetectedPlane> LiDARPlaneDetector<numRows>::detect(nrt::PointCloud2 & cloud)
{
  cloud.addField<Distance>();


  frame_timing::start("Clear Accumulator");
  itsAccumulator.clear();
  frame_timing::stop("Clear Accumulator");

  // Get distances to all points
  frame_timing::start("Compute Distances");
  for(auto i = cloud.begin<Distance>(), end = cloud.end<Distance>(); i != end; ++i)
    i->get<Distance>().value = i->geometry().getVector3Map().norm();
  frame_timing::stop("Compute Distances");

  // Find all of the groups in the cloud
  frame_timing::start("Find Groups");
  std::transform(itsRowRange.begin(), itsRowRange.end(), itsGroups.begin(),
      [&cloud](size_t row) { return findGroupsInRow<numRows>(cloud, row);});
  frame_timing::stop("Find Groups");

  // Allow each group to vote for all possible planes in the accumulator,
  // storing the cell id for any vote which passes threshold.
  std::vector<AccumulatorBall::cellid_t> allVotes;
  frame_timing::start("Voting");
  for(size_t row = 0; row<numRows; ++row)
    for(LiDARGroup & group : itsGroups[row])
    {
      std::vector<AccumulatorBall::cellid_t> groupVotes = voteForGroup(cloud, group, itsAccumulator, itsVoteSpread);
      allVotes.insert(allVotes.end(), groupVotes.begin(), groupVotes.end());
    }
  frame_timing::stop("Voting");

  // Ask the accumulator for the plane estimates in the set of accumulator
  // cells that passed threshold
  frame_timing::start("Vote Rendering");
  std::vector<DetectedPlane> detectedPlanes;
  for(AccumulatorBall::cellid_t cellid : allVotes)
  {
    AccumulatorBall::VoteCell voteCell = itsAccumulator.getVoteCell(cellid);
    DetectedPlane detectedPlane;
    detectedPlane.plane = voteCell.plane;
    detectedPlane.groups = voteCell.voters;
    detectedPlanes.push_back(detectedPlane);
  }
  frame_timing::stop("Vote Rendering");

  // Perform a set of filtering options to pair down and refine the number of
  // plane estimates
  std::map<std::string, size_t> numPlanes;
  numPlanes["1. start"] = detectedPlanes.size();
  frame_timing::start("Filtering");
  { // Remove any planes who were voted on by less than N lasers
    frame_timing::start("Planes 05a.  Filter Min Lasers");
    detectedPlanes = filterMinLasers<numRows>(detectedPlanes, 2);
    numPlanes["2. filter min lasers"] = detectedPlanes.size();
    frame_timing::stop("Planes 05a.  Filter Min Lasers");
  }

  { // Split planes by finding clusters in their groups
    frame_timing::start("Planes 05b.  Split Planes");
    detectedPlanes = splitPlanes<numRows>(detectedPlanes, itsGroups, cloud);
    numPlanes["3. split planes"] = detectedPlanes.size();
    frame_timing::stop("Planes 05b.  Split Planes");
  }

  { // Refine the planes by refitting them to the groups that voted for them
    frame_timing::start("Planes 05c.  Refining");
    std::transform(detectedPlanes.begin(), detectedPlanes.end(), detectedPlanes.begin(),
        std::bind(refinePlaneFromPoints<numRows>, _1, itsGroups, cloud, RefineFrom::Groups));
    frame_timing::stop("Planes 05c.  Refining");

    int count = 0;
    for( auto & p : detectedPlanes )
      for(auto gid : p.groups)
        count += groupFromId<numRows>(itsGroups, gid).indices.size();
    numPlanes["900. numPoints"] = count;
  }

  //report(numPlanes);

  { // Merge planes by clustering those that were voted for by the same groups
    frame_timing::start("Planes 05c.  Smart Merge");
    detectedPlanes = smartMerge<numRows>(detectedPlanes, itsGroups, cloud);
    numPlanes["4. smart merge"] = detectedPlanes.size();
    frame_timing::stop("Planes 05c.  Smart Merge");
  }
  frame_timing::stop("Filtering");

  // Now that we are done with groups, accumulate all of the indices into one
  // index list for each plane
  frame_timing::start("Index Rendering");
  for(DetectedPlane & plane : detectedPlanes)
    for(groupid_t gid : plane.groups)
    {
      LiDARGroup const & group = groupFromId(itsGroups, gid);
      plane.indices.insert(plane.indices.end(), group.indices.begin(), group.indices.end());
    }
  frame_timing::stop("Index Rendering");


  //CALLGRIND_START_INSTRUMENTATION;
  frame_timing::start("Region Growing");
  //detectedPlanes = growRegions2<numRows>(detectedPlanes, cloud);
  detectedPlanes = growRegions(detectedPlanes, itsGroups, cloud, 0.05, 10);
  frame_timing::stop("Region Growing");
  //CALLGRIND_STOP_INSTRUMENTATION;
  numPlanes["5. region growing"] = detectedPlanes.size();

  frame_timing::start("Polygonalization");
  for(DetectedPlane & plane : detectedPlanes)
    plane.hull = convexHullPolygonalization(plane, cloud);
  frame_timing::stop("Polygonalization");

  frame_timing::start("Polygon Merge");
  detectedPlanes = mergePolygons<numRows>(detectedPlanes, itsGroups, cloud,
      config::lookup<double>("planedetection.polygonmerge.dotProduct"),
      config::lookup<double>("planedetection.polygonmerge.rho"),
      config::lookup<double>("planedetection.polygonmerge.overlap"));
  frame_timing::stop("Polygon Merge");
  numPlanes["6. polygon merge"] = detectedPlanes.size();

  frame_timing::start("Least Squares Fit");
  std::transform(detectedPlanes.begin(), detectedPlanes.end(), detectedPlanes.begin(),
      std::bind(refinePlaneLeastSquares, _1,  cloud));
  frame_timing::stop("Least Squares Fit");

  //report(numPlanes);

  return detectedPlanes;
}

template class LiDARPlaneDetector<32>;
template class LiDARPlaneDetector<64>;
