#pragma once

#include <nrt/Eigen/Eigen.H>
#include <nrt/PointCloud2/PointCloud2.H>
#include "../DetectedPlane.H"
#include <boost/geometry.hpp>
#include <boost/geometry/geometries/point_xy.hpp>
#include <boost/geometry/geometries/polygon.hpp>

namespace polygonhelpers
{
  typedef boost::geometry::model::d2::point_xy<float> Point;
  typedef boost::geometry::model::polygon<Point> Polygon;

  // ######################################################################
  //! Find the polygon with the largest area from a list of polygons
  inline Polygon findLargestPolygon(std::vector<Polygon> const & polygons)
  {
    std::vector<float> areas(polygons.size());
    std::transform(polygons.begin(), polygons.end(), areas.begin(), boost::geometry::area<Polygon>);

    size_t idx = std::distance(areas.begin(), std::max_element(areas.begin(), areas.end()));

    return polygons[idx];
  }

  // ######################################################################
  //! Find the average plane from a list of detected planes
  inline Eigen::Hyperplane<float, 3> findAveragePlane(std::vector<DetectedPlane> const & componentPlanes)
  {
    Eigen::Vector3f avgNormal(0, 0, 0);
    float avgOffset = 0;

    for(DetectedPlane const & p : componentPlanes)
    {
      avgNormal += p.plane.normal();
      avgOffset += p.plane.offset();
    }

    return Eigen::Hyperplane<float, 3>(avgNormal / componentPlanes.size(), avgOffset / componentPlanes.size());
  }

  // ######################################################################
  //! Find the average plane from a list of planes
  inline Eigen::Hyperplane<float, 3> findAveragePlane(std::vector<Eigen::Hyperplane<float, 3>> const & componentPlanes)
  {
    Eigen::Vector3f avgNormal(0, 0, 0);
    float avgOffset = 0;

    for(auto const & p : componentPlanes)
    {
      avgNormal += p.normal();
      avgOffset += p.offset();
    }

    return Eigen::Hyperplane<float, 3>(avgNormal / componentPlanes.size(), avgOffset / componentPlanes.size());
  }

  // ######################################################################
  //! Project a 3D hull onto a 3D plane, and return the 2D projected polygon
  inline Polygon projectPolygon(std::vector<nrt::Point3D<float>> const & hull, Eigen::Hyperplane<float, 3> const & averagePlane)
  {
    // Get the rotation from the normal of the plane to the Z axis

    Eigen::Quaterniond rotation_d;
    rotation_d.setFromTwoVectors(averagePlane.normal().cast<double>(), Eigen::Vector3d::UnitZ());
    Eigen::Quaternionf rotation = rotation_d.cast<float>();

//    Eigen::Quaternionf rotation;
//    rotation.setFromTwoVectors(averagePlane.normal(), Eigen::Vector3f::UnitZ());

    // Transform the subset of points into 2D by rotating into the plane
    std::vector<Point> points2D(hull.size());
    std::transform(hull.begin(), hull.end(), points2D.begin(),
        [&rotation](nrt::Point3D<float> const & p3D)
        {
          Eigen::Vector3f const rotPoint = rotation * Eigen::Vector3f(p3D.x(), p3D.y(), p3D.z());
          return Point(rotPoint.x(), rotPoint.y());
        });

    Polygon projectedPolygon;
    points2D.push_back(points2D.front());
    boost::geometry::append(projectedPolygon, points2D);
    boost::geometry::correct(projectedPolygon);

    return projectedPolygon;
  }

  // ######################################################################
  //! Project the hull of a detected plane onto a 3D plane, and return the 2D projected polygon
  inline Polygon projectPlaneHull(DetectedPlane const & plane, Eigen::Hyperplane<float, 3> const & averagePlane)
  {
    return projectPolygon(plane.hull, averagePlane);
  }

  // ######################################################################
  //! Reproject a 2D polygon back onto a 3D plane, and return the resulting 3D polygon
  inline std::vector<nrt::Point3D<float>> reprojectPolygon(Polygon const & polygon2D, Eigen::Hyperplane<float, 3> const & plane)
  {
    Eigen::Quaterniond inverseRotation_d;
    inverseRotation_d.setFromTwoVectors(Eigen::Vector3d::UnitZ(), plane.normal().cast<double>());
    Eigen::Quaternionf inverseRotation = inverseRotation_d.cast<float>();
    //inverseRotation.setFromTwoVectors(Eigen::Vector3f::UnitZ(), plane.normal());

    // Transform the 2D hull back into 3D
    std::vector<nrt::Point3D<float>> hull3D(polygon2D.outer().size());
    std::transform(polygon2D.outer().begin(), polygon2D.outer().end(), hull3D.begin(),
        [plane, inverseRotation](Point const & p) {
        Eigen::Vector3f p3 = plane.projection(inverseRotation*Eigen::Vector3f(p.x(), p.y(), 0));
        return nrt::Point3D<float>(p3.x(), p3.y(), p3.z());
        });

    return hull3D;
  }

  // ######################################################################
  inline float intersectionOverUnion(DetectedPlane const & plane1, DetectedPlane const & plane2)
  {
    auto const hull1Transformed = plane1.hull;
    auto const hull2Transformed = plane2.hull;

    auto const averagePlane = polygonhelpers::findAveragePlane({plane1,plane2});
    auto const projectedHull1 = polygonhelpers::projectPolygon(hull1Transformed, averagePlane);
    auto const projectedHull2 = polygonhelpers::projectPolygon(hull2Transformed, averagePlane);

    std::vector<polygonhelpers::Polygon> intersections;
    boost::geometry::intersection(projectedHull1, projectedHull2, intersections);

    if(!intersections.empty())
    {
      std::vector<polygonhelpers::Polygon> unions;
      boost::geometry::union_(projectedHull1, projectedHull2, unions);

      if(!unions.empty())
      {
        float const i_area = boost::geometry::area(polygonhelpers::findLargestPolygon(intersections));
        float const u_area = boost::geometry::area(polygonhelpers::findLargestPolygon(unions));
        float const intersection_over_union = i_area / u_area;

        return intersection_over_union;
      }
    }
    return 0;
  }

  // ######################################################################
  //! Find the centroid of a hull. Note that this is not the same as the weighted average
  //! of all of the points - it is the actual center of mass of the 3D hull.
  inline nrt::Point3D<float> hullCentroid(std::vector<nrt::Point3D<float>> const & hull, Eigen::Hyperplane<float, 3> plane)
  {
    Polygon const projectedHull = projectPolygon(hull, plane);
    Point centroid2D(std::numeric_limits<float>::quiet_NaN(), std::numeric_limits<float>::quiet_NaN());
    boost::geometry::centroid(projectedHull, centroid2D);

    Eigen::Quaterniond inverseRotation_d;
    inverseRotation_d.setFromTwoVectors(Eigen::Vector3d::UnitZ(), plane.normal().cast<double>());
    Eigen::Quaternionf inverseRotation = inverseRotation_d.cast<float>();
    //inverseRotation.setFromTwoVectors(Eigen::Vector3f::UnitZ(), plane.normal());

    Eigen::Vector3f const centroid3Dv = plane.projection(inverseRotation * Eigen::Vector3f(centroid2D.x(), centroid2D.y(), 0));

    return nrt::Point3D<float>( centroid3Dv.x(), centroid3Dv.y(), centroid3Dv.z() );
  }

}

