#pragma once

#include <nrt/PointCloud2/PointCloud2.H>
#include <nrt/PointCloud2/Features/FeatureBase.H>
#include <nrt/PointCloud2/Features/FeatureTypes/PointNormal.H>

//! Compute the normals for every point in a Velodyne point cloud
/*! \note This method takes advantage of the ordering of Velodyne point clouds,
          so make sure that input has not been modified or filtered in any way.
    \param input The input cloud
    \param distance The distance in meters to collect neighbors from a square area.
    \param twoPass Whether to use the more stable but slower two pass or the faster but less stable one pass
                   covariance calculation */
nrt::PointCloud2 computeVelodyneNormals( nrt::PointCloud2 const input, float const distance, bool twoPass = false);

//! Computes the normals for points originating from a Velodyne HDL-32E
/*! This takes advantage of the structure of velodyne points to compute the normals without
    needing to create any search structure to find nearest neighbors.  Normals extracted in
    this fashion must therefor come from point clouds that have not had their structure
    modified or been filtered in any way.  Clouds that have been adjusted prior to this normal
    extraction will not receive correct normal estimates.

    This computes normals using total least squares. */
class VelodyneNormals : public nrt::FeatureBase
{
  public:
    //! Create a new VelodyneNormals object
    /*  @param distance The distance in meters to search for neighbors
        @param twoPass Whether to use the more numerically stable two-pass covariance calculation, or a faster but less stable one pass version
        @param viewPoint The viewpoint from which to calculate normals */
    VelodyneNormals( double distance,
                     bool twoPass = false,
                     nrt::PointCloud2::Geometry const & viewPoint = nrt::PointCloud2::Geometry() );

    //! Virtual destruction
    virtual ~VelodyneNormals();

    //! Finds normals for all points in the input
    /*! This will add the field PointNormals to the output point cloud */
    virtual nrt::PointCloud2 computeFeatures( nrt::PointCloud2 const input );

    //! Finds normals for all points in the subset
    /*! Neighbors will be considered from the entire cloud, not just the subset.  Valid normals will
        only be calculated for points in the subset.

        This will add the field PointNormals to the output point cloud */
    virtual nrt::PointCloud2 computeFeatures( nrt::PointCloud2 const input, nrt::Indices const subset );

    //! Gets the neighbors around a specific point in a cloud up to some distance away
    /*! @param cloud The cloud
        @param index The point in the cloud to use
        @param distance The search distance.  This is not a radius but a square patch around the query point
        @return The set of indices corresponding to the neighbors */
    static nrt::Indices velodyneNeighbors( nrt::PointCloud2 const cloud, int const index, double distance );

    static nrt::Indices velodynePatch( nrt::PointCloud2 const cloud, int const idx, int const dr, int const dc);

  protected:

    //! Does the actual work of finding the features
    void computeFeaturesImpl( nrt::PointCloud2 const & cloud,
                              nrt::PointCloud2::Iterator<nrt::PointNormal> && iter,
                              nrt::PointCloud2::Iterator<nrt::PointNormal> && end );
  private:
    double itsDistance;
    bool itsUseTwoPassCovariance;
    nrt::PointCloud2::Geometry itsViewPoint;
};
