/*! @file
    @author Randolph Voorhies
    @copyright GNU Public License (GPL v3)
    @section License
    @verbatim
    // ////////////////////////////////////////////////////////////////////////
    //              The iLab Neuromorphic Robotics Toolkit (NRT)             //
    // Copyright 2010-2012 by the University of Southern California (USC)    //
    //                          and the iLab at USC.                         //
    //                                                                       //
    //                iLab - University of Southern California               //
    //                Hedco Neurociences Building, Room HNB-10               //
    //                    Los Angeles, Ca 90089-2520 - USA                   //
    //                                                                       //
    //      See http://ilab.usc.edu for information about this project.      //
    // ////////////////////////////////////////////////////////////////////////
    // This file is part of The iLab Neuromorphic Robotics Toolkit.          //
    //                                                                       //
    // The iLab Neuromorphic Robotics Toolkit is free software: you can      //
    // redistribute it and/or modify it under the terms of the GNU General   //
    // Public License as published by the Free Software Foundation, either   //
    // version 3 of the License, or (at your option) any later version.      //
    //                                                                       //
    // The iLab Neuromorphic Robotics Toolkit is distributed in the hope     //
    // that it will be useful, but WITHOUT ANY WARRANTY; without even the    //
    // implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       //
    // PURPOSE.  See the GNU General Public License for more details.        //
    //                                                                       //
    // You should have received a copy of the GNU General Public License     //
    // along with The iLab Neuromorphic Robotics Toolkit.  If not, see       //
    // <http://www.gnu.org/licenses/>.                                       //
    // ////////////////////////////////////////////////////////////////////////
    @endverbatim */

#include "PlaneConstraint.H"
#include <nrt/Eigen/EigenDecomposition.H>

// ######################################################################
PlaneConstraint::PlaneConstraint(double threshold) :
  threshold_(threshold), m_(decltype(m_)::Zero()), dirty_(true)
{ }

// ######################################################################
void PlaneConstraint::setThreshold(double threshold)
{ threshold_ = threshold; }

// ######################################################################
double PlaneConstraint::getThreshold()
{ return threshold_; }

// ######################################################################
void PlaneConstraint::addConstraint(Eigen::Vector3f const & normal, double const weight)
{
  Eigen::Vector3f const n = normal.normalized();

  m_(0,0) += n.x()*n.x() * weight;
  m_(1,1) += n.y()*n.y() * weight;
  m_(2,2) += n.z()*n.z() * weight;

  m_(1,0) += n.y()*n.x() * weight;
  m_(2,0) += n.z()*n.x() * weight;
  m_(2,1) += n.z()*n.y() * weight;

  dirty_ = true;
}

// ######################################################################
void PlaneConstraint::computeSVD()
{
  if(dirty_)
  {
    m_(0,1) = m_(1,0);
    m_(0,2) = m_(2,0);
    m_(1,2) = m_(2,1);

    //std::cout << "Constraints: " << std::endl << m_ << std::endl;

    Eigen::JacobiSVD<decltype(m_)> svd(m_, Eigen::ComputeFullV);

    auto s = svd.singularValues();

    R_t_ = svd.matrixV().transpose();
    radii_ = s.cwiseSqrt();

    dirty_ = false;
  }
}

// ######################################################################
Eigen::Vector3d PlaneConstraint::radii()
{
  computeSVD();
  return radii_;
}

// ######################################################################
Eigen::Matrix3d PlaneConstraint::rotation()
{
  computeSVD();
  return R_t_.transpose();
}

// ######################################################################
Eigen::Matrix3d PlaneConstraint::constraints()
{
  return m_;
}

// ######################################################################
double PlaneConstraint::getScore(Eigen::Vector3f const & normal)
{
  computeSVD();

  Eigen::Vector3f const n = normal.normalized();

  // Solve the polar-coordinate ellipsoid equation to find the radius in the direction
  // of the query normal
  auto qr = R_t_ * n.cast<double>();

  auto const theta = std::atan(qr[1] / qr[0]);
  auto const phi   = std::acos(qr[2] / qr.norm());

  auto const stheta = std::sin(theta);
  auto const ctheta = std::cos(theta);
  auto const sphi   = std::sin(phi);
  auto const cphi   = std::cos(phi);

  auto const r = std::sqrt( 1.0 /
      (
       ctheta*ctheta * sphi*sphi / (radii_[0]*radii_[0]) +
       stheta*stheta * sphi*sphi / (radii_[1]*radii_[1]) +
       cphi*cphi / (radii_[2]*radii_[2])
      ));

  return 1.0 - std::min(r/threshold_, 1.0);
}
