#include "IOUtils.H"
#include <pcl/conversions.h>
#include <pcl/point_cloud.h>
#include <pcl/point_types.h>
#include "pcl_conversions.h"
#include <nrt/Core/Image/PixelTypes.H>
#include <PointCloud/Features/RangeImage.H>
#include <PointCloud/Features/VelodyneHelpers.H>
#include <nrt/PointCloud2/Common/Transforms.H>

#include <rosbag/bag.h>
#include <sensor_msgs/PointCloud2.h>

struct CloseEnough
{
  bool operator() (float a, float b) { return std::abs(a-b) > 0.1 && a < b; };
};

int main(int argc, char ** argv)
{
  assert(argc == 3);
  std::string filename(argv[1]);
  std::string ext = filename.substr(filename.find_last_of(".")+1);

  DataSource const datasource =
    (ext == "nrtlog") ? DataSource::nrtlog :
#ifdef USE_MRPT
    (ext == "rawlog") ? DataSource::mrptlog :
#endif // USE_MRPT
    DataSource::nrtlog;

  VelodyneIO reader( filename, datasource );

  rosbag::Bag bag(argv[2], rosbag::bagmode::Write);
  ros::Time rostime;

  int frame = 1;
  reader.getCloud( frame++ );


  while(true)
  {
    try
    {
      std::cout << "Frame: " << frame << std::endl;
      nrt::Optional<nrt::PointCloud2> nrt_cloud = reader.getCloud( frame++ );
      if(!nrt_cloud) { std::cout << "No Cloud available... " << std::endl; break; }

      //nrt::transformPointCloudInPlace(*nrt_cloud,
      //    Eigen::Affine3f(Eigen::AngleAxisf(M_PI/2.0, -Eigen::Vector3f::UnitX())));

      pcl::PointCloud<pcl::PointXYZI> pcl_cloud;
      pcl_cloud.reserve(nrt_cloud->size());

      std::set<float, CloseEnough> angles;


      std::vector<size_t> rows = {
        16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31
      };

      bool const use_color = nrt_cloud->hasField<nrt::PixRGB<uint8_t>>();

      for(size_t col=0; col<nrt_cloud->size()/32; col+=1)
      {
        for(size_t row : rows)
        {
          size_t const index = velodyne32::index(row,col);
          nrt::PointCloud2::Geometry const & g = (*nrt_cloud)[index];
          float const intensity = use_color ?
            nrt_cloud->get<nrt::PixRGB<uint8_t>>(index).get<nrt::PixRGB<uint8_t>>().r()/255.0F :
            1.0;
          //auto const & p = nrt_cloud->get<nrt::PixRGB<uint8_t>>(index);
          //auto const & g = p.geometry();

          pcl::PointXYZI ret;
          ret.x = g.x();
          ret.y = g.y();
          ret.z = g.z();
          ret.intensity = intensity;

          if(!g.isValid() || std::sqrt(ret.x*ret.x + ret.z*ret.z) <= 0.0) continue;

          pcl_cloud.push_back(ret);


          pcl::PointXYZ mix;
          mix.x = ret.y;
          mix.y = ret.z;
          mix.z = ret.x;

          float const angle = std::atan(mix.y / std::sqrt(mix.x*mix.x + mix.z*mix.z)) * 180/M_PI;
          angles.insert(angle);

          int scanID = int(0.75 * angle + 0.5) + 7;
          if (angle < 0) {
            scanID--;
          }
          if(scanID < 0 || scanID >= 16)
          {
          float const angle2 = std::atan(g.z() / std::sqrt(g.x()*g.x() + g.y()*g.y())) * 180/M_PI;
          std::cerr << "Bad point! angle: " << angle << " scanid: " << scanID << " (angle2: " << angle2 << ")" << std::endl;
          }
          assert(scanID >= 0 && scanID < 16);

        }
      }

      std::cout << "Angles: " << std::endl;
      for(auto angle : angles)
      {
        int scanID = int(0.75 * angle + 0.5) + 7;
        if (angle < 0) {
          scanID--;
        }

        std::cout << angle << " ::: " << scanID << std::endl;
        assert(scanID >= 0 && scanID < 16);
      }


      auto time = rostime.fromSec(frame*0.1);

      sensor_msgs::PointCloud2 pcl_cloud_msg;
      pcl::toROSMsg(pcl_cloud, pcl_cloud_msg);
      pcl_cloud_msg.header.stamp = time;
      pcl_cloud_msg.header.frame_id = "/velodyne";


      bag.write("/velodyne_cloud", time, pcl_cloud_msg);
    }
    catch(...) { std::cout << "Caught Exception!" << std::endl; break; }
  }
  bag.close();
}
