#include "IOUtils.H"
#include <nrt/Core/Model/Manager.H>
#include <nrt/Graphics/ShapeRendererBasic.H>
#include <nrt/Graphics/Shapes.H>
#include <nrt/PointCloud2/Common/Transforms.H>
#include <Config/Config.H>
#include <LiDARPlaneDetector.H>
#include <DrawingUtils/DrawingUtils.H>
#include <PointCloud/Features/VelodyneNormalsFLAS.H>

#include <nrt/ImageProc/IO/ImageSink/VideoWriters/FfmpegVideoWriter.H>
#include <nrt/ImageProc/Reshaping/Transform.H>

using namespace nrt;

NRT_DECLARE_PARAMETER(filename, std::string,  "The data file or image directory", "/home/rand/drc/messagelogs/hedco0.nrtlog");
NRT_DECLARE_PARAMETER(frame,    int,          "The starting frame number", 1);
NRT_DECLARE_PARAMETER(autozoom, bool,         "Zoom in a sine wave", false);
NRT_DECLARE_PARAMETER(alpha,    float,        "Zoom alpha", 0.01);
NRT_DECLARE_PARAMETER(beta,     float,        "Zoom beta", 30);
NRT_DECLARE_PARAMETER(configfile,       std::string,  "Path to the config file", "../config/default.cfg");
struct Parameters : public nrt::Component, public nrt::Parameter<
                    filename, frame, alpha, beta, configfile, autozoom
                    >
{ using nrt::Component::Component; };

int main( int argc, char const ** argv )
{
  Manager mgr(argc, argv);

  auto renderer = mgr.addComponent<graphics::ShapeRendererBasic>("Renderer");
  auto parameters = mgr.addComponent<Parameters>("Parameters");

  mgr.launch();

  config::openconfig(parameters->configfile::get());

  int frame = parameters->frame::get();

  std::string ext = parameters->filename::get().substr(parameters->filename::get().find_last_of(".")+1);
  DataSource const datasource = 
    (ext == "nrtlog") ? DataSource::nrtlog :
#ifdef USE_MRPT
    (ext == "rawlog") ? DataSource::mrptlog :
#endif // USE_MRPT
    DataSource::nrtlog;

  nrt::FfmpegVideoWriter videoWriter;
  std::string const filebasename = parameters->filename::get().substr(0, parameters->filename::get().find_last_of('.'));
  videoWriter.open(filebasename + ".mpeg");
  std::cout << "Writing " << filebasename + ".mpeg" << std::endl;

  VelodyneIO reader( parameters->filename::get(), datasource );

  LiDARPlaneDetector<32> planeDetector(
      config::lookup<double>("planedetection.thetaBins"),
      config::lookup<double>("planedetection.phiBins"),
      config::lookup<double>("planedetection.rhoBins"),
      70, 0.1, false);
  planeDetector.setAccumulatorThreshold(config::lookup<double>("planedetection.accumulatorThreshold"));//2.0);

  VelodyneNormalsFLAS normalsEngine(3,1); // theta, phi
  while( true )
  {
    renderer->initFrame();

    if(parameters->autozoom::get())
    {
      float const pos = (std::sin(frame*parameters->alpha::get())+1.0)/2.0*parameters->beta::get() + 5.0;
      renderer->lookAt(Eigen::Vector3d(0,-pos,pos), Eigen::Vector3d(0,0,0), Eigen::Vector3d::UnitZ());
    }

    auto cloud = reader.getCloud( frame++ );
    normalsEngine.compute( *cloud );
    graphics::PointCloud cloudGraphics(*cloud);

    std::vector<std::shared_ptr<nrt::graphics::Shape>> shapes;
    drawPlanes(shapes, planeDetector.detect(*cloud), Eigen::Isometry3d::Identity(), 200);
    for(auto s : shapes) s->render(*renderer);

    cloudGraphics.render(*renderer);

    renderer->renderFrame();

    nrt::Image<nrt::PixRGB<uint8_t>> framebuffer(renderer->dims());
    glReadPixels(0,0,framebuffer.width(),framebuffer.height(), GL_RGB, GL_UNSIGNED_BYTE, framebuffer.pod_begin());
    framebuffer = nrt::flipVertical(framebuffer);
    videoWriter.appendFrame(nrt::GenericImage(framebuffer));

  }

  return 0;
}


