#define COMPILE_ERROR_NOT_IMPLEMENTED

#include <nrt/Eigen/Eigen.H>
#include "PlaneSLAM.H"
#include <gtsam/geometry/Pose3.h>
#include <gtsam/geometry/Point3.h>
#include <gtsam/slam/PriorFactor.h>
#include <gtsam/slam/BetweenFactor.h>
#include <gtsam/nonlinear/Values.h>
#include <gtsam/nonlinear/Symbol.h>
#include <gtsam/linear/linearExceptions.h>
#include <boost/lexical_cast.hpp>
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/connected_components.hpp>

#include <PointCloud/Features/Planes/details/FrameTiming.H>
#include <PointCloud/Features/Planes/details/PolygonHelpers.H>

#include <Config/Config.H>

#include "PoseToPlaneFactor.H"
#include "PolygonUtils.H"
#include "Util.H"

#include <valgrind/callgrind.h>


std::string planeslam::getEigenVersion()
{
  return std::to_string(EIGEN_WORLD_VERSION) + "."
       + std::to_string(EIGEN_MAJOR_VERSION) + "."
       + std::to_string(EIGEN_MINOR_VERSION);
}

gtsam::NonlinearFactorGraph global_graph;

#if GTSAM_VERSION_MAJOR == 2

using Vector4 = gtsam::Vector_;
using Vector6 = gtsam::Vector_;

gtsam::Vector_ makeVector4(double v1, double v2, double v3, double v4)
{ return gtsam::Vector_(4, v1, v2, v3, v4); }

gtsam::Vector_ makeVector6(double v1, double v2, double v3, double v4, double v5, double v6)
{ return gtsam::Vector_(6, v1, v2, v3, v4, v5, v6); }

#elif GTSAM_VERSION_MAJOR == 3

using Vector4 = Eigen::Matrix<double, 4, 1>;
using Vector6 = Eigen::Matrix<double, 6, 1>;

Vector4 makeVector4(double v1, double v2, double v3, double v4)
{ return (Vector4() << v1, v2, v3, v4); }

Vector6 makeVector6(double v1, double v2, double v3, double v4, double v5, double v6)
{ return (Vector6() << v1, v2, v3, v4, v5, v6); }

#elif GTSAM_VERSION_MAJOR == 4

gtsam::Vector makeVector4(double v1, double v2, double v3, double v4)
{ return (gtsam::Vector(4) << v1, v2, v3, v4).finished(); }

gtsam::Vector makeVector6(double v1, double v2, double v3, double v4, double v5, double v6)
{ return (gtsam::Vector(6) << v1, v2, v3, v4, v5, v6).finished(); }
#endif

// ######################################################################
PlaneSLAM::PlaneSLAM() :
  itsISAM(createParameters()),
  itsPoseIndex(0),
  itsMap(),
  itsInitialized(false),
  itsMinObservations(5)
{ }

// ######################################################################
gtsam::ISAM2Params PlaneSLAM::createParameters() const
{
  gtsam::ISAM2Params params;
  //params.relinearizeThreshold = 0.00001;
  //params.evaluateNonlinearError = true;
  //params.relinearizeSkip = 1;
  //params.factorization = gtsam::ISAM2Params::QR;
  return params;
}

// ######################################################################
nrt::Optional<gtsam::Symbol> matchPlaneToMap(Eigen::Affine3f const & currPose, DetectedPlane & plane,
    PlaneSLAM::RenderedMap const & map)
{
  Eigen::Hyperplane<float, 3> p1 = plane.plane;
  p1.transform(currPose);
  std::vector<nrt::Point3D<float>> h1 = plane.hull;
  std::transform(h1.begin(), h1.end(), h1.begin(), [currPose](nrt::Point3D<float> const & p)
      {
      auto pt = currPose * Eigen::Vector3f(p.x(), p.y(), p.z());
      return nrt::Point3D<float>(pt.x(), pt.y(), pt.z());
      });
  Eigen::Vector3f const c1 = p1.projection(currPose * plane.centroid->cast<float>());

  float const minDotProduct1           = std::cos(config::lookup<double>("planeslam.mapMatch.minDotProduct1")*M_PI/180);
  float const maxOffsetDifference1     = config::lookup<double>("planeslam.mapMatch.maxOffsetDifference1");
  float const minIntersectionOverUnion = config::lookup<double>("planeslam.mapMatch.minIntersectionOverUnion");

  float const minDotProduct2         = std::cos(config::lookup<double>("planeslam.mapMatch.minDotProduct2")*M_PI/180);
  float const maxOffsetDifference2   = config::lookup<double>("planeslam.mapMatch.maxOffsetDifference2");
  float const minIntersectionOverMin = config::lookup<double>("planeslam.mapMatch.minIntersectionOverMin");

  nrt::Optional<gtsam::Symbol> bestSymbol = nrt::OptionalEmpty;

  gtsam::Symbol bestSymbol1;
  float bestScore1 = -std::numeric_limits<float>::infinity();
  float offsetDiff1 = 0;
  float dotProduct1 = 0;

  gtsam::Symbol bestSymbol2;
  float bestScore2 = -std::numeric_limits<float>::infinity();
  float offsetDiff2 = 0;
  float dotProduct2 = 0;

  for(auto const & mapHullPair : map)
  {
    auto const & mapHull = mapHullPair.second;
    Eigen::Hyperplane<float, 3> const & p2 = mapHull.plane;
    std::vector<nrt::Point3D<float>> const & h2 = mapHull.hull;
    Eigen::Vector3f c2 = p2.projection(Eigen::Vector3f(mapHull.centroid.x(), mapHull.centroid.y(), mapHull.centroid.z()));

    const double normalDot  = p1.normal().dot(p2.normal());

    if(normalDot < minDotProduct1) continue;

    auto const averagePlane = polygonhelpers::findAveragePlane({p1,p2});

    const double offsetDiff = std::abs( (c1 - c2).dot(averagePlane.normal()) );
    if(offsetDiff >= maxOffsetDifference1) continue;

    auto const projectedHull1 = polygonhelpers::projectPolygon(h1, averagePlane);
    auto const projectedHull2 = polygonhelpers::projectPolygon(h2, averagePlane);

    std::vector<polygonhelpers::Polygon> intersections;
    try
    {
      boost::geometry::intersection(projectedHull1, projectedHull2, intersections);
    }
    catch(boost::geometry::overlay_invalid_input_exception e)
    {
      std::cerr << "##############################################################" << std::endl;
      std::cerr << "##############################################################" << std::endl;
      std::cerr << "##############################################################" << std::endl;
      std::cerr << "Caught Exception in " << __FILE__ << " : " << __FUNCTION__ << ":" << __LINE__ << std::endl;
      std::cerr << e.what() << std::endl;
      std::cerr << "projectedHull1.size() : " << projectedHull1.outer().size() << std::endl;
      std::cerr << "projectedHull2.size() : " << projectedHull2.outer().size() << std::endl;
      std::cerr << "##############################################################" << std::endl;
      std::cerr << "##############################################################" << std::endl;
      std::cerr << "##############################################################" << std::endl;
      //throw e;
    }

    if(!intersections.empty())
    {
      std::vector<polygonhelpers::Polygon> unions;
      boost::geometry::union_(projectedHull1, projectedHull2, unions);

      if(!unions.empty())
      {
        float const i_area = boost::geometry::area(polygonhelpers::findLargestPolygon(intersections));
        float const u_area = boost::geometry::area(polygonhelpers::findLargestPolygon(unions));
        float const score = i_area / u_area;

        if(i_area > 0.0 && u_area > 0.0 && score > bestScore1)
        {
          bestScore1 = score;
          bestSymbol1 = mapHull.symbol;
          offsetDiff1 = offsetDiff;
          dotProduct1 = normalDot;
        }
      }
    }

    // Case 2 computations
    if(offsetDiff >= maxOffsetDifference2 || normalDot < minDotProduct2) continue;

    if(!intersections.empty())
    {
      float const i_area  = boost::geometry::area(polygonhelpers::findLargestPolygon(intersections));
      float const p1_area = boost::geometry::area(projectedHull1);
      float const p2_area = boost::geometry::area(projectedHull2);

      float const score = i_area / std::min(p1_area, p2_area);

      if(i_area > 0.0 && p1_area > 0.0 && p2_area > 0.0 && score > bestScore2)
      {
        bestScore2 = score;
        bestSymbol2 = mapHull.symbol;
        offsetDiff2 = offsetDiff;
        dotProduct2 = normalDot;
      }
    }
  } // end frame2 loop

  if(bestScore1 > minIntersectionOverUnion)
  {
    plane.debugMessage = " : Matched 1! " + std::to_string(bestScore1) + " " + std::to_string(offsetDiff1) + " " + std::to_string(dotProduct1);
    return bestSymbol1;
  }
  else if(bestScore2 > minIntersectionOverMin)
  {
    plane.debugMessage = " : Matched 2! " + std::to_string(bestScore2) + " " + std::to_string(offsetDiff2) + " " + std::to_string(dotProduct2);
    return bestSymbol2;
  }

  return nrt::OptionalEmpty;
}

// ######################################################################
template <class FactorType>
void addToGraph(gtsam::NonlinearFactorGraph & graph, FactorType const & factor)
{
  graph.add(factor);
  global_graph.add(factor);
}

  // ######################################################################
void PlaneSLAM::organizeSymbols(Eigen::Affine3f const & currPose, gtsam::Symbol const & currPoseSymbol,
    nrt::Correspondences const & correspondences, std::vector<DetectedPlane> const & lastPlanes,
    std::vector<DetectedPlane> & currentPlanes)
{
  frame_timing::guard timer("Organize Symbols");

  // Push any symbol information through correspondences
  for(nrt::Correspondence const & corr : correspondences)
  {
    assert(corr.sourceIndex < int(lastPlanes.size()));
    assert(corr.targetIndex < int(currentPlanes.size()));

    DetectedPlane const & last = lastPlanes[corr.sourceIndex];
    DetectedPlane & curr       = currentPlanes[corr.targetIndex];

    curr.pastObservations = last.pastObservations;
    curr.symbol           = last.symbol;
    curr.accepted         = last.accepted;
    if(!curr.accepted) curr.pastObservations.emplace_back(currPoseSymbol, last);
  }

  // For any planes without a symbol, try to find a match in the global map
  auto map = getRenderMap(); // TODO: cache this
  for(DetectedPlane & plane : currentPlanes)
  {
    if(!plane.symbol)
    {
      plane.symbol = matchPlaneToMap(currPose, plane, map);
      if(plane.symbol) plane.accepted = true;
    }
  }
}

// ######################################################################
void PlaneSLAM::globalMerge( std::vector<DetectedPlane> & currentPlanes )
{
  frame_timing::guard timer("Global Merge");

  auto map = getRenderMap();

  typedef boost::adjacency_list<boost::setS, boost::vecS, boost::undirectedS> Graph;
  Graph graph;

  float const minDotProduct1           = std::cos(config::lookup<double>("planeslam.globalMerge.minDotProduct1")*M_PI/180);
  float const maxOffsetDifference1     = config::lookup<double>("planeslam.globalMerge.maxOffsetDifference1");
  float const minIntersectionOverUnion = config::lookup<double>("planeslam.globalMerge.minIntersectionOverUnion");

  float const minDotProduct2         = std::cos(config::lookup<double>("planeslam.globalMerge.minDotProduct2")*M_PI/180);
  float const maxOffsetDifference2   = config::lookup<double>("planeslam.globalMerge.maxOffsetDifference2");
  float const minIntersectionOverMin = config::lookup<double>("planeslam.globalMerge.minIntersectionOverMin");

  std::map<size_t, gtsam::Symbol> idx2symbol;
  std::map<gtsam::Symbol, size_t> symbol2idx;
  size_t idx = 0;
  for(auto p : map)
  {
    idx2symbol[idx] = p.second.symbol;
    symbol2idx[p.second.symbol] = idx;
    ++idx;
  }

  // Create graph where planes are linked if they are similar enough
  std::vector<gtsam::Symbol> symbols;
  for(RenderedMap::iterator it_i = map.begin(); it_i != map.end(); ++it_i)
  {
    symbols.push_back(it_i->second.symbol);

    auto const & p1 = it_i->second.plane;
    auto const & h1 = it_i->second.hull;
    Eigen::Vector3f const c1(it_i->second.centroid.x(), it_i->second.centroid.y(), it_i->second.centroid.z());
    size_t const i = symbol2idx[it_i->second.symbol];

    for(RenderedMap::iterator it_j = it_i; it_j != map.end(); ++it_j)
    {
      auto const & p2 = it_j->second.plane;
      auto const & h2 = it_j->second.hull;
      Eigen::Vector3f const c2(it_j->second.centroid.x(), it_j->second.centroid.y(), it_j->second.centroid.z());
      size_t const j = symbol2idx[it_j->second.symbol];

      if(i == j)
      {
        boost::add_edge(i, j, graph);
        continue;
      }

      float const normalDot = p1.normal().dot(p2.normal());

      if(normalDot < minDotProduct1) continue;

      auto const averagePlane = polygonhelpers::findAveragePlane({p1,p2});

      float const offsetDiff = std::abs( (c1 - c2).dot(averagePlane.normal()) );

      if(offsetDiff > maxOffsetDifference1) continue;

      auto const projectedHull1 = polygonhelpers::projectPolygon(h1, averagePlane);
      auto const projectedHull2 = polygonhelpers::projectPolygon(h2, averagePlane);

      std::vector<polygonhelpers::Polygon> intersections;
      try
      {
        boost::geometry::intersection(projectedHull1, projectedHull2, intersections);
      }
      catch(boost::geometry::overlay_invalid_input_exception e)
      {
        std::cerr << "##############################################################" << std::endl;
        std::cerr << "##############################################################" << std::endl;
        std::cerr << "##############################################################" << std::endl;
        std::cerr << "Caught Exception in " << __FILE__ << " : " << __FUNCTION__ << ":" << __LINE__ << std::endl;
        std::cerr << e.what() << std::endl;
        std::cerr << "projectedHull1.size() : " << projectedHull1.outer().size() << std::endl;
        std::cerr << "projectedHull2.size() : " << projectedHull2.outer().size() << std::endl;
        std::cerr << "##############################################################" << std::endl;
        std::cerr << "##############################################################" << std::endl;
        std::cerr << "##############################################################" << std::endl;
        //throw e;
      }

      if(intersections.empty()) continue;

      std::vector<polygonhelpers::Polygon> unions;
      boost::geometry::union_(projectedHull1, projectedHull2, unions);

      if(unions.empty()) continue;

      float const i_area = boost::geometry::area(polygonhelpers::findLargestPolygon(intersections));
      if(i_area <= 0.0) continue;

      float const u_area = boost::geometry::area(polygonhelpers::findLargestPolygon(unions));
      if(u_area <= 0.0) continue;

      if(i_area / u_area > minIntersectionOverUnion)
      {
        boost::add_edge(i, j, graph);
      }
      else
      {
        if(offsetDiff > maxOffsetDifference2 || normalDot < minDotProduct2) continue;

        float const p1_area = boost::geometry::area(projectedHull1);
        float const p2_area = boost::geometry::area(projectedHull2);

        if(i_area / std::min(p1_area, p2_area) > minIntersectionOverMin)
        {
          boost::add_edge(i, j, graph);
        }
      }
    }
  }

  // Find all of the connected components in the graph and cluster them together
  std::vector<size_t> components(boost::num_vertices(graph));
  int numComponents = boost::connected_components(graph, components.data());

  // build list of component indices
  std::vector<std::vector<gtsam::Symbol>> componentList(numComponents);
  for( size_t index = 0; index < components.size(); ++index )
    componentList[components[index]].push_back(idx2symbol[index]);

  // For each set of merged planes (vector in componentList),
  // merge them together somehow...

  gtsam::NonlinearFactorGraph factorUpdatesGraph;
  gtsam::FastVector<size_t> indicesToRemove;

  // Get the current factor graph from isam
  gtsam::NonlinearFactorGraph const & factorGraph = itsISAM.getFactorsUnsafe();
  // Get the variable index map from isam
  gtsam::VariableIndex const & variableIndex = itsISAM.getVariableIndex();
  // Get the ordering
#if GTSAM_VERSION_MAJOR == 2
  auto const & ordering = itsISAM.getOrdering();
#endif

  for( std::vector<gtsam::Symbol> const & componentSymbols : componentList )
  {
    if( componentSymbols.size() < 2 )
      continue;

    // get the symbol of the first component
    gtsam::Symbol symbol = componentSymbols.front();
    auto & metadata = itsMap[symbol];

    // for all other components, remove them from the graph
    // and update their factors to use the new symbol
    for( size_t i = 1; i < componentSymbols.size(); ++i )
    {
      // Find all of the factors that are connected to component[i]
      auto oldSymbol = componentSymbols[i];

#if GTSAM_VERSION_MAJOR == 2
      auto factors = variableIndex[ordering[oldSymbol]];
#elif GTSAM_VERSION_MAJOR == 3
      gtsam::FastList<size_t> const & factors = variableIndex[oldSymbol];
#elif GTSAM_VERSION_MAJOR == 4
      auto const & factors = variableIndex[oldSymbol];
#endif
      std::cout << "Combining " << std::string(oldSymbol) << " into " << std::string(symbol) << std::endl;

      // Combine hull
      auto const & oldMetadata = itsMap[oldSymbol];
      metadata.insert(metadata.end(), oldMetadata.begin(), oldMetadata.end());
      // remove old meta data
      itsMap.erase( itsMap.find(oldSymbol) );


      // For each factor this component touches, rekey it and delete it
      for( size_t const & f : factors )
      {
        // Add a rekeyed factor to the graph
        factorUpdatesGraph.add( factorGraph[f]->rekey( {{{oldSymbol, symbol}}} ));

        // Remove the original factor from the graph
        indicesToRemove.push_back(f);
      }

      // Update all symbols in current planes
      for( auto & plane : currentPlanes )
        plane.symbol.ifSet( [&](gtsam::Symbol s){ if( s == oldSymbol ) plane.symbol = symbol; } );
    }
  }

#if GTSAM_VERSION_MAJOR == 2
  // update with new values and deletions
  itsISAM.update(factorUpdatesGraph, gtsam::Values(), indicesToRemove);
#elif GTSAM_VERSION_MAJOR >= 3
  // In GTSAM 3.1, the removal doesn't work (it just causes an exception to be thrown). The fix is to
  // replace the factors-to-be-removed with huge covariances
  for(size_t const i : indicesToRemove)
  {
    double hugenum = 10000000000.0;
    gtsam::noiseModel::Diagonal::shared_ptr hugeNoise =
      gtsam::noiseModel::Diagonal::Variances(makeVector4(hugenum, hugenum, hugenum, hugenum));

    boost::shared_ptr<PoseToPlaneFactor> f = boost::dynamic_pointer_cast<PoseToPlaneFactor>(factorGraph[i]);
    assert(f);
    factorUpdatesGraph.add(PoseToPlaneFactor( f->measurement(), hugeNoise, f->poseKey(), f->planeKey() ));
  }
  itsISAM.update(factorUpdatesGraph);
#endif
}

// ######################################################################
std::vector<Eigen::Isometry3d> PlaneSLAM::update(Eigen::Isometry3d  odometry,
    nrt::Correspondences const & correspondences,
    std::vector<DetectedPlane> const & lastPlanes,
    std::vector<DetectedPlane> & currentPlanes,
    Eigen::Matrix3d const & translationCovariance, Eigen::Matrix3d const & rotationCovariance)
{
  frame_timing::guard timer("Update");

  if(!itsInitialized) { initializeGraph(odometry); return {odometry}; }

  //CALLGRIND_START_INSTRUMENTATION;

  auto const o_c_r = 0.03;// + std::pow(0.001, 2);  // Odometry rotation
  auto const o_c_t = 0.01;// + std::pow(0.001, 2);  // Odometry translation
  auto const p_c_n = 0.01;//std::pow(0.05, 2);  // Plane normal
  auto const p_c_o = 0.05;//std::pow(0.1, 2);  // Plane offset

  // Compose odometry covariance
  Eigen::Matrix<double, 6, 6> odometryCovariance = Eigen::Matrix<double, 6, 6>::Zero();
  odometryCovariance.block<3,3>(0,0) = translationCovariance;
  odometryCovariance.block<3,3>(3,3) = rotationCovariance;

  // Baseline uncertainty - also helps prevent numerical errors
  //odometryCovariance += Eigen::Matrix<double, 6, 6>::Identity() * std::pow(0.01, 2);

  //gtsam::noiseModel::Base::shared_ptr odometryNoise;

  //auto const odometryCovNorm = odometryCovariance.norm();
  //if(odometryCovNorm == 0 || std::isnan(odometryCovNorm) || odometryCovNorm > 10.0 || odometry.translation().norm() > 0.5) // baseline if calculated covariance was invalid
  //{
  //  std::cout << "Bad odometry covariance" << std::endl;
  //  odometryNoise = gtsam::noiseModel::Diagonal::Variances(
  //      makeVector6(o_c_r, o_c_r, o_c_r, o_c_t, o_c_t, o_c_t)
  //      );
  //  odometry = Eigen::Isometry3d::Identity();
  //}
  //else
  //{
  //  //odometryCovariance.block<3,3>(0,0) *= (1 + odometry.translation().norm());
  //  odometryNoise = gtsam::noiseModel::Gaussian::Covariance(odometryCovariance);
  //}
  static const double cauchyK = config::lookup<double>("planeslam.cauchyK");

  auto odometryNoise =
    gtsam::noiseModel::Robust::Create(gtsam::noiseModel::mEstimator::Cauchy::Create(cauchyK),
        gtsam::noiseModel::Diagonal::Variances(makeVector6(o_c_r, o_c_r, o_c_r, o_c_t, o_c_t, o_c_t))
        );

  //std::cout << odometryCovariance << std::endl;

  // Compose plane covariance
  auto measurementNoise =
    gtsam::noiseModel::Robust::Create(gtsam::noiseModel::mEstimator::Cauchy::Create(cauchyK),
        gtsam::noiseModel::Diagonal::Variances(makeVector4(p_c_n, p_c_n, p_c_n, p_c_o))
        );

  static int planeSymbolIdx = 0;

  gtsam::Symbol lastPoseSymbol('x', itsPoseIndex-1);
  gtsam::Symbol currPoseSymbol('x', itsPoseIndex++);

  gtsam::NonlinearFactorGraph graph;
  gtsam::Values initialEstimate;

  // Get the last pose estimate from iSAM, and the current one by applying odometry to it
  gtsam::Pose3 lastPoseEstimate = itsISAM.calculateEstimate<gtsam::Pose3>(lastPoseSymbol);
  gtsam::Pose3 currPoseEstimate = gtsam::Pose3(odometry.matrix() * lastPoseEstimate.matrix());
  Eigen::Affine3f currPoseAffine(currPoseEstimate.matrix().cast<float>());

  // Add the odometry factor to the graph
  addToGraph(graph, gtsam::BetweenFactor<gtsam::Pose3>(lastPoseSymbol, currPoseSymbol, gtsam::Pose3(odometry.matrix()), odometryNoise));
  initialEstimate.insert(currPoseSymbol, currPoseEstimate);

  // Organize the symbols (push through, or match to global) of currentPlanes
  organizeSymbols(currPoseAffine, currPoseSymbol, correspondences, lastPlanes, currentPlanes);

  // Insert the planes into the graph
  for(DetectedPlane & plane : currentPlanes)
  {
    // Plane is already being tracked and this is a new matched observation of it
    if(plane.accepted)
    {
      addToGraph(graph, PoseToPlaneFactor( PlaneValue( plane.plane.cast<double>() ), measurementNoise, currPoseSymbol, *(plane.symbol) ) );
      itsMap[*(plane.symbol)].emplace_back( currPoseSymbol, plane.hull );
    }
    // this is a plane we've tracked for enough iterations
    else if(plane.pastObservations.size() > itsMinObservations)
    {
      // This is a plane we haven't seen before but have successfully tracked for enough iterations
      assert(!plane.symbol);
      plane.symbol = gtsam::Symbol('z', planeSymbolIdx++);

      // Calculate the initial estimate for this plane, which is in global coordinates
      //  Transform using current pose (global -> local) inverse, which is (local -> global)
      Eigen::Hyperplane<double, 3> globalPlaneEstimate = plane.plane.cast<double>().transform( Eigen::Affine3d(currPoseEstimate.inverse().matrix()) );
      initialEstimate.insert(*plane.symbol, PlaneValue(globalPlaneEstimate));

      // insert all of the tracked observations that led to passing threshold
      for(auto & old_obs : plane.pastObservations)
      {
        addToGraph(graph, PoseToPlaneFactor( PlaneValue( old_obs.plane->plane.cast<double>() ), measurementNoise, old_obs.poseSymbol, *(plane.symbol) ) );
        itsMap[*(plane.symbol)].emplace_back( old_obs.poseSymbol, old_obs.plane->hull );
      }

      plane.accepted = true;
      plane.pastObservations.clear();
    }
  }

  try
  {
  itsISAM.update(graph, initialEstimate);
  }
  catch(gtsam::IndeterminantLinearSystemException const & e)
  {
    std::cout << "OH SHIT: " << e.what() << " variable: " << e.nearbyVariable() << std::endl;
  }
  globalMerge( currentPlanes );

  //// Save current graph to disk
  //saveGraphGEXF("./graph.gexf", global_graph, getMap());

  // Collect current pose estimates
  std::vector<Eigen::Isometry3d> poses;
  for(int i = 0; i < itsPoseIndex; ++i)
    poses.emplace_back(itsISAM.calculateEstimate<gtsam::Pose3>(gtsam::Symbol('x', i)).matrix());

  //CALLGRIND_STOP_INSTRUMENTATION;
  return poses;
}

// ######################################################################
void PlaneSLAM::initializeGraph(Eigen::Isometry3d const & odometry)
{
  gtsam::NonlinearFactorGraph graph;
  gtsam::Values initialEstimate;

  gtsam::Symbol currPoseSymbol('x', itsPoseIndex++);

  gtsam::noiseModel::Diagonal::shared_ptr priorNoise =
    gtsam::noiseModel::Diagonal::Variances(makeVector6( 1e-9, 1e-9, 1e-9, 1e-9, 1e-9, 1e-9));

  addToGraph(graph, gtsam::PriorFactor<gtsam::Pose3>(currPoseSymbol, gtsam::Pose3(odometry.matrix()), priorNoise));

  initialEstimate.insert(currPoseSymbol, gtsam::Pose3(odometry.matrix()));

  itsISAM.update(graph, initialEstimate);

  itsInitialized = true;
}

// ######################################################################
std::map<gtsam::Symbol, Eigen::Hyperplane<float, 3>> PlaneSLAM::getMap()
{
  std::map<gtsam::Symbol, Eigen::Hyperplane<float, 3>> map;

  gtsam::Values estimateValues = itsISAM.calculateEstimate();

  for(auto keyValue : estimateValues.filter<PlaneValue>())
  {
    gtsam::Symbol symbol(keyValue.key);
    Eigen::Hyperplane<float, 3> plane(keyValue.value.n().cast<float>(), keyValue.value.d());
    map.emplace(symbol, plane);
  }

  return map;
}

// ######################################################################
auto PlaneSLAM::getRenderMap() -> RenderedMap
{
  frame_timing::guard timer("Render Map");
  RenderedMap renderedMap;

  gtsam::Values estimateValues = itsISAM.calculateEstimate();

  size_t cached_entries = 0;
  size_t non_cached_entries = 0;
  //std::cout << "---------------------------" << std::endl;
  for( const auto & mapEntry : itsMap )
  {
    // Look up name and plane
    const gtsam::Symbol name( mapEntry.first );

    // Get the current estimated value of the plane from ISAM
    const auto planeValue = estimateValues.at<PlaneValue>( mapEntry.first );
    Eigen::Hyperplane<float, 3> plane(planeValue.n().cast<float>(), planeValue.d());

    auto const & metadata = mapEntry.second;

    size_t const numHulls = metadata.size();

    // Search the cache for an entry
    auto cacheEntry = itsRenderMapCache.find(name);
    if(cacheEntry != itsRenderMapCache.end())
    {
      auto const & cacheHull = cacheEntry->second.mapHull;
      Eigen::Hyperplane<float,3> cachePlane = cacheHull.plane;
      std::size_t cacheNumHulls = cacheEntry->second.numHulls;

      // If the cache entry has the same humber of hulls, and the exact same plane then we can just use
      // the cache instead of recomputing the hull unions.
      if((numHulls == cacheNumHulls) && (cachePlane.offset() == plane.offset()) && (cachePlane.normal() == plane.normal()))
      {
        ++cached_entries;
        renderedMap.emplace(cacheHull.symbol, cacheHull);
        //renderedMap[cacheHull.symbol] = cacheHull;
        continue;
      }
    }
    ++non_cached_entries;

    // Get all hulls for this plane
    std::vector<std::vector<nrt::Point3D<float>>> hulls;
    hulls.reserve( numHulls );

    // For each hull associated with the plane, transform it to the global coordinate frame
    // according to its current pose estimate
    for( const auto & meta : metadata )
    {
      auto const & pose = estimateValues.at<gtsam::Pose3>( meta.pose ).matrix();
      auto hull = meta.hull;

      for( auto & p : hull )
      {
        const auto v = pose * Eigen::Vector4d( p.x(), p.y(), p.z(), 1.0 );
        p = nrt::Point3D<float>(v.x(), v.y(), v.z());
      }

      hulls.emplace_back( hull );
    }

    // Merge all of the hulls into a convex hull
    RenderedMapHull mergedHull(name, plane, mergeHullsSimple(plane, hulls));

    //// Merge all of the hulls into a non-convex hull
    //RenderedMapHull mergedHull(name, plane, mergeHulls(plane, hulls));

    renderedMap[mergedHull.symbol] = mergedHull;

    // TODO: Need to clean up cache when hulls get merged!
    itsRenderMapCache[name] = RenderMapCacheEntry( mergedHull, numHulls );
  }
  // note: crisp

  //std::cout << "Cached/Non-Cached: " << cached_entries << "/" << non_cached_entries << std::endl;
  return renderedMap;
}
