#pragma once
#include <gtsam/nonlinear/ISAM2.h>
#include <DetectedPlane.H>
#include <nrt/PointCloud2/Common/Correspondence.H>
#include <gtsam/nonlinear/NonlinearFactorGraph.h>
#include "details/PolygonHelpers.H"

namespace planeslam
{
  std::string getEigenVersion();
}

class PlaneSLAM
{
  public:
    PlaneSLAM();

    std::map<gtsam::Symbol, Eigen::Hyperplane<float, 3>> getMap();

    std::vector<Eigen::Isometry3d> update(Eigen::Isometry3d odometry,
        nrt::Correspondences const & correspondences,
        std::vector<DetectedPlane> const & lastPlanes, std::vector<DetectedPlane> & currentPlanes,
        Eigen::Matrix3d const & translationCovariance, Eigen::Matrix3d const & rotationCovariance);

    struct PlaneMetaData
    {
      PlaneMetaData() = default;
      PlaneMetaData( gtsam::Symbol s, std::vector<nrt::Point3D<float>> const & h ) :
        pose( s ), hull( h )
      { }

      gtsam::Symbol pose;
      std::vector<nrt::Point3D<float>> hull;
    };

    //! A rendered 
    struct RenderedMapHull
    {
      RenderedMapHull() { }

      RenderedMapHull(
          gtsam::Symbol const & symbol,
          Eigen::Hyperplane<float,3> const & plane, 
          std::vector<nrt::Point3D<float>> hull,
          nrt::Optional<nrt::Point3D<float>> centroid = nrt::OptionalEmpty) :
        symbol(symbol),
        plane(plane),
        hull(hull),
        centroid( (centroid ? *centroid : polygonhelpers::hullCentroid(hull, plane)) )
        { }

      gtsam::Symbol symbol;                  //! The symbol (z?) of this plane
      Eigen::Hyperplane<float,3> plane;      //! The infinite plane
      std::vector<nrt::Point3D<float>> hull; //! The 3D hull
      nrt::Point3D<float> centroid;          //! The center of mass of the 3D hull
    };

    using RenderedMap = std::map<gtsam::Symbol, RenderedMapHull>;

    RenderedMap getRenderMap();

  private:
    void organizeSymbols(Eigen::Affine3f const & currPose, gtsam::Symbol const & currPoseSymbol,
        nrt::Correspondences const & correspondences, std::vector<DetectedPlane> const & lastPlanes,
        std::vector<DetectedPlane> & currentPlanes);

    void initializeGraph(Eigen::Isometry3d const & odometry);

    void globalMerge( std::vector<DetectedPlane> & currentPlanes);

    gtsam::ISAM2Params createParameters() const;
    gtsam::ISAM2 itsISAM;

    int itsPoseIndex;

    std::map<gtsam::Key, std::vector<PlaneMetaData>> itsMap;
    bool itsInitialized;

    size_t itsMinObservations;

    struct RenderMapCacheEntry
    {
      RenderMapCacheEntry() {};
      RenderMapCacheEntry(RenderedMapHull const & m, size_t const n) : mapHull(m), numHulls(n) {}

      RenderedMapHull mapHull;
      size_t numHulls;
    };
    std::map<gtsam::Symbol, RenderMapCacheEntry> itsRenderMapCache;

};
