#include "PolygonUtils.H"
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/connected_components.hpp>
#include <boost/geometry.hpp>
#include <boost/geometry/geometries/point_xy.hpp>
#include <boost/geometry/geometries/polygon.hpp>
#include <boost/geometry/multi/geometries/multi_point.hpp>
#include <fstream>

using namespace std::placeholders;

typedef boost::geometry::model::d2::point_xy<float> Point;
typedef boost::geometry::model::polygon<Point> Polygon;
typedef boost::adjacency_list<boost::setS, boost::vecS, boost::undirectedS> Graph;

namespace
{
  // ######################################################################
  // Project a polygon onto a plane, returning the 2D result
  std::vector<Point> projectPoints(std::vector<nrt::Point3D<float>> const & hull, Eigen::Hyperplane<float, 3> const & plane)
  {
    // Get the rotation from the normal of the plane to the Z axis
    Eigen::Quaternionf rotation;
    rotation.setFromTwoVectors(plane.normal(), Eigen::Vector3f::UnitZ());

    // Transform the subset of points into 2D by rotating into the plane
    std::vector<Point> points2D(hull.size());
    std::transform(hull.begin(), hull.end(), points2D.begin(),
        [&rotation](nrt::Point3D<float> const &  p3D)
        {
          Eigen::Vector3f const rotPoint = rotation * Eigen::Vector3f(p3D.x(), p3D.y(), p3D.z());
          return Point(rotPoint.x(), rotPoint.y());
        });
    return points2D;
  }


  // ######################################################################
  // Project a polygon onto a plane, returning the 2D boost.geometry.Polygon result
  //   This is just a wrapper around projectPoints
  Polygon projectPolygon(std::vector<nrt::Point3D<float>> const & hull, Eigen::Hyperplane<float, 3> const & plane)
  {
    if(hull.empty()) return Polygon();

    std::vector<Point> points2D = projectPoints(hull, plane);

    Polygon projectedPolygon;
    points2D.push_back(points2D.front());
    boost::geometry::append(projectedPolygon, points2D);
    boost::geometry::correct(projectedPolygon);

    return projectedPolygon;
  }

  // ######################################################################
  // Reproject a 2D polygon from a plane back into 3D
  std::vector<nrt::Point3D<float>> reprojectPolygon(Polygon const & polygon2D, Eigen::Hyperplane<float, 3> const & plane)
  {
    Eigen::Quaternionf inverseRotation;
    inverseRotation.setFromTwoVectors(Eigen::Vector3f::UnitZ(), plane.normal());

    // Transform the 2D hull back into 3D
    std::vector<nrt::Point3D<float>> hull3D(polygon2D.outer().size());
    std::transform(polygon2D.outer().begin(), polygon2D.outer().end(), hull3D.begin(),
        [plane, inverseRotation](Point const & p) {
        Eigen::Vector3f p3 = plane.projection(inverseRotation*Eigen::Vector3f(p.x(), p.y(), 0));
        return nrt::Point3D<float>(p3.x(), p3.y(), p3.z());
        });

    return hull3D;
  }
}

// ######################################################################
float overlapRatio(std::vector<nrt::Point3D<float>> const & h1, Eigen::Hyperplane<float, 3> const & p1,
    std::vector<nrt::Point3D<float>> const & h2, Eigen::Hyperplane<float, 3> const & p2,
    nrt::Optional<float&> h1AreaOutput, nrt::Optional<float&> h2AreaOutput)
{
  if(h1.size() < 3 || h2.size() < 3) return 0;

  Eigen::Hyperplane<float, 3> averagePlane((p1.normal() + p2.normal()).normalized(), (p1.offset() + p2.offset()) / 2.0);

  Polygon polygon1 = projectPolygon(h1, averagePlane);
  Polygon polygon2 = projectPolygon(h2, averagePlane);

  std::deque<Polygon> intersections;
  try
  {
    boost::geometry::intersection(polygon1, polygon2, intersections);
  }
  catch(...)
  {
    return 0;
  }

  if(intersections.empty()) return 0.0;

  float intersectionArea = 0.0;
  for(Polygon const & i : intersections)
  {
    assert(boost::geometry::area(i) > 0);
    intersectionArea += boost::geometry::area(i);
  }

  float const polygon1Area = boost::geometry::area(polygon1);
  float const polygon2Area = boost::geometry::area(polygon2);

  if(h1AreaOutput) *h1AreaOutput = polygon1Area;
  if(h2AreaOutput) *h2AreaOutput = polygon2Area;

  return intersectionArea / std::min(polygon1Area, polygon2Area);
}

//// ######################################################################
//std::vector<std::vector<nrt::Point3D<float>>> mergeHulls(
//    Eigen::Hyperplane<float, 3> const & plane, std::vector<std::vector<nrt::Point3D<float>>> const & hulls)
//{
//  if(hulls.size() == 0) return {};
//  if(hulls.size() == 1) return {hulls[0]};
//
//  // 1. Project all of the hulls onto the plane
//  std::vector<Polygon> projectedPolygons(hulls.size());
//  std::transform(hulls.begin(), hulls.end(), projectedPolygons.begin(), std::bind(projectPolygon, _1, plane));
//
//  // 2. Create a graph with each element of the current component as a node.
//  //    Edges between elements exist if the projected polygons have a non-zero
//  //    intersection
//  Graph componentGraph;
//  for(size_t i=0; i<hulls.size(); ++i)
//    for(size_t j=i; j<hulls.size(); ++j)
//    {
//      if(boost::geometry::intersects(projectedPolygons[i], projectedPolygons[j]))
//        boost::add_edge(i, j, componentGraph);
//    }
//
//  // 3. Find all of the connected components in the graph and assemble them into componentPolygons
//  std::vector<size_t> polygonComponents(boost::num_vertices(componentGraph));
//  size_t numPolyComponents = boost::connected_components(componentGraph, polygonComponents.data());
//  std::vector<std::vector<Polygon>> componentPolygons(numPolyComponents);
//  assert(projectedPolygons.size() == polygonComponents.size());
//  for(size_t i=0; i<projectedPolygons.size(); ++i)
//  {
//    componentPolygons[polygonComponents[i]].push_back(projectedPolygons[i]);
//  }
//  assert(numPolyComponents > 0);
//
//  // 4. Create a set of non-convex hulls around each subcomponent by finding the union
//  //    of all member polygons
//  std::vector<Polygon> unions;
//  for(size_t subcomponent=0; subcomponent<numPolyComponents; ++subcomponent)
//  {
//    Polygon unionPolygon;
//    for(Polygon const & poly : componentPolygons[subcomponent])
//    {
//      std::deque<Polygon> unions;
//      boost::geometry::union_(unionPolygon, poly, unions);
//      // TODO: Sometimes we get more than one result here - need to investigate
//      unionPolygon = unions[0];
//    }
//    Polygon simplifiedUnion;
//    boost::geometry::simplify(unionPolygon, simplifiedUnion, .01);
//    unions.push_back(simplifiedUnion);
//  }
//  assert(unions.size() > 0);
//
//  // 5. Reproject the hulls into 3D
//  std::vector<std::vector<nrt::Point3D<float>>> mergedHulls(unions.size());
//  std::transform(unions.begin(), unions.end(), mergedHulls.begin(), std::bind(reprojectPolygon, _1, plane));
//
//  return mergedHulls;
//}

// ######################################################################
std::vector<nrt::Point3D<float>> mergeHulls(
    Eigen::Hyperplane<float, 3> const & plane, std::vector<std::vector<nrt::Point3D<float>>> const & hulls)
{
  if(hulls.size() == 0) return {};
  if(hulls.size() == 1) return {hulls[0]};

  std::vector<Polygon> projectedPolygons(hulls.size());
  std::transform(hulls.begin(), hulls.end(), projectedPolygons.begin(), std::bind(projectPolygon, _1, plane));

  Polygon unionPolygon;
  for(Polygon const & p : projectedPolygons)
  {
    std::deque<Polygon> unions;
    boost::geometry::union_(unionPolygon, p, unions);

    if(unions.size() == 0)
    {
      continue;
    }
    else if(unions.size() == 1)
    {
      unionPolygon = unions[0];
    }
    else
    {
      std::vector<double> unionSizes(unions.size());
      std::transform(unions.begin(), unions.end(), unionSizes.begin(), [](Polygon const & p) {return boost::geometry::area(p);});
      unionPolygon = unions[std::distance(unionSizes.begin(), std::max_element(unionSizes.begin(), unionSizes.end()))];
    }
  }

  Polygon simplifiedUnionPolygon;
  boost::geometry::simplify(unionPolygon, simplifiedUnionPolygon, 0.25);

  std::vector<nrt::Point3D<float>> hull3D = reprojectPolygon(simplifiedUnionPolygon, plane);

  return {hull3D};
}

// ######################################################################
std::vector<nrt::Point3D<float>> mergeHullsSimple(
    Eigen::Hyperplane<float, 3> const & plane, std::vector<std::vector<nrt::Point3D<float>>> const & hulls)
{
  if(hulls.size() == 0) return {};
  if(hulls.size() == 1) return {hulls[0]};

  std::vector<std::vector<Point>> projectedPointSets(hulls.size());
  std::transform(hulls.begin(), hulls.end(), projectedPointSets.begin(), std::bind(projectPoints, _1, plane));

  boost::geometry::model::multi_point<Point> allPoints;
  for(std::vector<Point> p : projectedPointSets)
    boost::geometry::append(allPoints, p);

  Polygon convexHull;
  boost::geometry::convex_hull(allPoints, convexHull);

  std::vector<nrt::Point3D<float>> hull3D = reprojectPolygon(convexHull, plane);

  return {hull3D};
}
