#include "AccumulatorBall.H"

namespace 
{
  constexpr double rad (double deg) { return deg * M_PI  / 180.0; }
  constexpr double deg (double rad) { return rad * 180.0 / M_PI;  }
}

// ######################################################################
AccumulatorBall::AccumulatorBall(size_t const nTheta, size_t const nPhi, size_t const nRho, float const rhoMax, float const rhoSigma) :
  itsNumPhi(nPhi), itsNumRho(nRho), itsRhoMax(rhoMax),
  itsRhoStep(rhoMax / static_cast<float>(nRho)), itsPhiStep((M_PI / 2.0) / nPhi),
  itsRhoSigma(rhoSigma), itsRhoGaussianN1(1.0 / (rhoSigma * sqrt(2.0*M_PI))), itsRhoGaussianN2(2.0*rhoSigma*rhoSigma),
  itsRhoNbins(((itsRhoSigma*M_PI/180.0) * 3 / itsRhoStep)*2+1)
{
  // 1. Build a table which tells us how many theta bins are at each phi azimuth.
  itsNumThetas.resize(itsNumPhi);
  double step = 180.0 / ((unsigned int)itsNumPhi);

  double const h_0 = std::cos(rad(90 - step));
  double const max_a = (2.0*M_PI * h_0) / float(nTheta);
  double const h_top = max_a / (2.0*M_PI);
  double const phi_top_rad = std::acos(1 - h_top);
  double const phi_top_deg = deg(phi_top_rad);
  step = (180.0 - 2.0 * phi_top_deg) / (itsNumPhi - 2.0);

  //   1a. Ensure that the top and bottoms of the sphere have a single cap
  itsNumThetas[0] = 1;
  itsNumThetas[itsNumPhi - 1] = 1;

  //   1b. Fill out the table.
  int counter = 1;
  for(float phi = phi_top_deg; phi < 90; phi += step, ++counter)
  {
    double const h_i = cos(rad(phi)) - cos(rad(phi + step));
    double const a_i = 2.0*M_PI * h_i;

    int const n_t = a_i / max_a;

    itsNumThetas[counter] = n_t;
    itsNumThetas[itsNumPhi - 1 - counter] = n_t;
  }

  itsThetaNormalizers.resize(itsNumPhi);
  itsThetaSteps.resize(itsNumPhi);
  for(size_t phiIndex=0; phiIndex<itsNumPhi; ++phiIndex)
  {
    itsThetaNormalizers[phiIndex] = (itsNumThetas[phiIndex] * 0.9999999999f) / (2.0f*float(M_PI));
    itsThetaSteps[phiIndex] = M_PI / float(itsNumThetas[phiIndex]);
  }
  
  itsRhoNormalizer = itsNumRho / itsRhoMax;
  itsPhiNormalizer = itsNumPhi *  0.9999999f / M_PI;

  // 2. Set up the google hash map, giving it very unlikely values for the empty and deleted keys.
  //    The load factor here means that no memory will be freed when we clear.
  itsAccumulator.set_empty_key(std::numeric_limits<cellid_t>::max());
  itsAccumulator.set_deleted_key(std::numeric_limits<cellid_t>::max() - 1);
  itsAccumulator.min_load_factor(0.0);
  
  // 3. Clear ourselves to get ready to roll.
  clear();
}

namespace 
{
  inline float calcThreshold(float const origThreshold, float const rho)
  { return std::min(std::max(0.0f, origThreshold - rho/2.0f), 3.0f); }
}

// ######################################################################
nrt::Optional<AccumulatorBall::cellid_t> AccumulatorBall::accumulate(float theta, float phi, float rho, vote_t vote, size_t voterId)
{
  if(rho >= itsRhoMax) return nrt::OptionalEmpty;
  
  // Get the indices of the center of the cell, pretending that we have a dense accumulator.
  int const rhoIndex = rho * itsRhoNormalizer;
  int const phiIndex = phi * itsPhiNormalizer;
  int const thetaIndex = theta * itsThetaNormalizers[phiIndex];
  
  float threshold = calcThreshold(itsThreshold, rho);
 
  return accumulateOnIndices(thetaIndex, phiIndex, rhoIndex, vote, voterId, threshold);
}

// ######################################################################
nrt::Optional<AccumulatorBall::cellid_t> AccumulatorBall::accumulateOnIndices(size_t thetaIndex, size_t phiIndex, size_t rhoIndex, vote_t vote, size_t voterId, float threshold)
{
  // Get the 1 dimensional ID of the accumulator cell for our flat accumulator map.
  cellid_t cellid = rhoIndex + itsNumRho*phiIndex + itsNumRho*itsNumPhi*thetaIndex;
  
  // Check to see if the cell already exists in the sparse accumulator
  AccumulatorType::iterator cellIterator = itsAccumulator.find(cellid);

  //   If not, then we need to create it
  if(cellIterator == itsAccumulator.end())
    cellIterator = itsAccumulator.insert(std::make_pair(cellid, VoteCell())).first;
  
  VoteCell & voteCell = cellIterator->second;
  
  // Make sure the voter has never voted for this cell before
  if(std::find(voteCell.voters.begin(), voteCell.voters.end(), voterId) != voteCell.voters.end())
    return nrt::OptionalEmpty;
  
  voteCell.vote += vote;
  voteCell.voters.push_back(voterId);
  
  if(voteCell.used) return nrt::OptionalEmpty;
  
  if(voteCell.vote > itsThreshold)
  {
    voteCell.used = true;
    return cellid;
  }
  
  return nrt::OptionalEmpty;
}

namespace 
{
  class Gaussian
  {
    public:
      Gaussian(float const stddev) :
        stddev(stddev), N1(1.0 / (stddev * sqrt(2.0*M_PI))), N2(2.0*stddev*stddev) { }

      float pdf(float const x)
      { return N1 * exp(- x*x / N2); }

    private:
      float mean, stddev, N1, N2;
  };
}

// ######################################################################
std::vector<AccumulatorBall::cellid_t> AccumulatorBall::accumulateSpread(float theta, float phi, float rho, vote_t vote, size_t voterId)
{
  std::vector<cellid_t> successfulVotes;
  
  int const phiCenterIdx = phi * itsPhiNormalizer;
  int const thetaCenterIdx = theta * itsThetaNormalizers[phiCenterIdx];
  
  int const rhoCenterIdx = rho * itsRhoNormalizer;
  float const rhoCenter = rhoCenterIdx*itsRhoStep + itsRhoStep/2.0;
  size_t const rhoHalfWidth = itsRhoNbins/2;
  
  for(size_t rIdx = 0; rIdx < itsRhoNbins; ++rIdx)
  {
    // Calculate the rho strength
    int const rhoOffset = rIdx - rhoHalfWidth;
    int const currRhoIdx = rhoCenterIdx + rhoOffset;
    float const currRho = currRhoIdx*itsRhoStep + itsRhoStep/2;
    if(currRho < 0) continue;
    if(currRho > itsRhoMax) continue;
    float const rhoDiff = (currRho - rhoCenter);
    float const rhoStrength =  itsRhoGaussianN1 * exp(- rhoDiff*rhoDiff / itsRhoGaussianN2); 

    float const voteStrength = vote * rhoStrength / itsRhoGaussianN1;

    float threshold = calcThreshold(itsThreshold, rho);
  
    nrt::Optional<cellid_t> voteResult = accumulateOnIndices(thetaCenterIdx, phiCenterIdx, currRhoIdx, voteStrength, voterId, threshold);
    if(voteResult) successfulVotes.push_back(*voteResult);
  }
  
  return successfulVotes;
}

// ######################################################################
void AccumulatorBall::clear()
{
  itsAccumulator.clear_no_resize();
}
  
// ######################################################################
void AccumulatorBall::setThreshold(vote_t threshold)
{
  itsThreshold = threshold;
}

// ######################################################################
AccumulatorBall::VoteCell AccumulatorBall::getVoteCell(cellid_t cellId)
{
  // Grab the VoteCell from the accumulator
  VoteCell & cell = itsAccumulator[cellId];

  // Compute the hyperplane for this cell
  size_t const thetaIndex = cellId / (itsNumRho * itsNumPhi);
  size_t const rhoIndex = (cellId - thetaIndex * itsNumRho * itsNumPhi) % itsNumRho;
  size_t const phiIndex = (cellId - rhoIndex - thetaIndex * itsNumRho * itsNumPhi) / itsNumRho;
  
  int const nTheta = itsNumThetas[phiIndex];

  float const phiCell   = (0.5f + phiIndex)   * (M_PI / itsNumPhi);
  float const thetaCell = (0.5f + thetaIndex) * (2.0f*M_PI / nTheta);
  float const rhoCell   = (0.5f + rhoIndex)   * itsRhoMax / itsNumRho;

  Eigen::Vector3f n(cos(thetaCell)*sin(phiCell), sin(thetaCell)*sin(phiCell), cos(phiCell));
  n = -n;
  n.normalize();
  
  cell.plane = Eigen::Hyperplane<float, 3>(n, rhoCell);
  
  return cell;
}

// ######################################################################
float AccumulatorBall::getRhoStep() const
{
  return itsRhoStep;
}
