#pragma once
#include <nrt/Eigen/Eigen.H>
#include <sparsehash/dense_hash_map>
#include <nrt/Core/Design/Optional.H>

//! A Hough accumulator designed to store votes for infinite planes parameterized by a [theta, phi, rho] triplet.
/*! The accumulator space is designed to be spherical such that each cell
  occupies a similar amount of metric space. This helps alleviate
  singularities brought on by the minimal parameterization. This code is
  heavily modified from that found in 3DTK's randomized hough transform
  (slam6d.sourceforge.net), and is explained in Borrman, et al 2011.*/
class AccumulatorBall
{
  public:
    //! A vote in the accumulator.
    typedef float vote_t;

    //! A unique id for a cell in the accumulator.
    typedef size_t cellid_t;

    //! A cell in the accumulator.
    struct VoteCell
    {
      //! Default constructor.
      VoteCell() : vote(0) {}

      //! The plane defined by the center of the cell.
      Eigen::Hyperplane<float, 3> plane;

      //! A list of all voters who have voted for this cell.
      std::vector<size_t> voters;

      //! The cummulative strength of the vote in this cell.
      float vote;

      //! Has the ID for this vote cell been returned from a call to accumulate() yet?
      /*! The end user has no use for this value, and should just ignore it. */
      bool used;
    };

    //! Construct a new accumulator. This allocates a lookup table for the non-uniform theta dimension.
    AccumulatorBall(size_t const nTheta, size_t const nPhi, size_t const nRho, float const rhoMax, float const rhoSigma=0.02);

    //! Add a vote to the accumulator.
    /*! @param [in] theta The theta (x/y plane) dimension
      @param [in] phi The phi (azimuthal) dimension
      @param [in] rho The rho (distance) dimension
      @param [in] vote The strength of the vote.
      @param [in] voterId An id representing the id of the entity casting the
      vote. This is used to make sure that no entity can cast
      more than one vote for an accumulator cell.
      @returns An optional cellid_t, which if set means that the latest vote
      caused the cell indicated by the cellid_t to pass threshold.

      For example,
      @code
      nrt::Optional<cellid_t> vote_success = myAccumulator.accumulate(t, p, r, 1.0, 200);
      if(vote_success) cellid_t cellid = *vote_success;
      @endcode
      */
    nrt::Optional<cellid_t> accumulate(float theta, float phi, float rho, vote_t vote, size_t voterId);

    std::vector<cellid_t> accumulateSpread(float theta, float phi, float rho, vote_t vote, size_t voterId);

    //! Clear the accumulator, setting all bins to zero.
    /*! Note that this invalidates all cellid_t's previously returned by accumulate() */
    void clear();

    //! Set the accumulator threshold.
    /*! i.e. the minimum cummulative vote strength for a cell before
     * accumulate() returns a success. */
    void setThreshold(vote_t threshold);

    //! Get a VoteCell from the accumulator.
    VoteCell getVoteCell(cellid_t cellid);

    float getRhoStep() const;

  private:
    
    nrt::Optional<cellid_t> accumulateOnIndices(size_t thetaIdx, size_t phiIdx, size_t rhoIdx, vote_t vote, size_t voterId, float threshold);

    std::vector<size_t> itsNumThetas;
    std::vector<float> itsThetaSteps;
    std::vector<size_t> itsThetaNormalizers;
    float itsRhoNormalizer;
    float itsPhiNormalizer;
    size_t const itsNumPhi;
    size_t const itsNumRho;
    float const itsRhoMax;
    float const itsRhoStep;
    float const itsPhiStep;
    
    float itsRhoSigma;
    float itsRhoGaussianN1;
    float itsRhoGaussianN2;
    float itsRhoVoteNormalizer;
    size_t itsRhoNbins;
    
    float itsThreshold;

    typedef google::dense_hash_map<cellid_t, VoteCell> AccumulatorType;
    AccumulatorType itsAccumulator;
};
