#include "PlaneAssociation.H"
#include "FindTransforms.H"
#include "details/LinearAlgebra.H"
#include "details/PolygonHelpers.H"
#include "details/BoostGeometry.H"
#include <Config/Config.H>
#include "details/FrameTiming.H"

NRT_BEGIN_UNCHECKED_INCLUDES;
#include <Eigen/Eigenvalues>
NRT_END_UNCHECKED_INCLUDES;

#include <set>

// ######################################################################
nrt::Correspondences findSimpleCorrespondences(
    std::vector<DetectedPlane> const & frame1Planes, std::vector<DetectedPlane> const & frame2Planes,
    float const maxAngle, float const maxRhoDifference)
{
  nrt::Correspondences correspondences;

  float const minDotProduct = cos(maxAngle*M_PI/180);

  for(size_t i=0; i<frame1Planes.size(); ++i)
  {
    size_t bestIdx = 0;
    float bestScore = std::numeric_limits<float>::infinity();

    Eigen::Hyperplane<float, 3> const & p1 = frame1Planes[i].plane;

    for(size_t j=0; j<frame2Planes.size(); ++j)
    {
      Eigen::Hyperplane<float, 3> const & p2 = frame2Planes[j].plane;
      if(std::abs(p1.offset()-p2.offset()) < maxRhoDifference && p1.normal().dot(p2.normal()) > minDotProduct)
      {
        float const score = (p1.normal()*p1.offset() - p2.normal()*p2.offset()).squaredNorm();
        if(score < bestScore)
        {
          bestIdx = j;
          bestScore = score;
        }
      }
    }

    if(bestScore != std::numeric_limits<float>::infinity())
      correspondences.emplace_back(i, bestIdx, bestScore);
  }

  return correspondences;
}

// ######################################################################
/*
 * Case 1: Hulls look nearly the same
 *  High intersection over union ratio when projected onto average plane.
 *  Allows us to work with loose offset and normal constraints
 *
 * Case 2: Some overlap
 *  Area of intersection is greater than some percentage of the area of
 *  smaller hull.
 *
 *  Needs more strict offset, possibly normal constraints.
 */
nrt::Correspondences findGoodCorrespondences(std::vector<DetectedPlane> const & frame1Planes, std::vector<DetectedPlane> & frame2Planes)
{
  nrt::Correspondences correspondences;

  using Hull = std::vector<nrt::Point3D<float>>;

  float const minDotProduct1 = std::cos(config::lookup<double>("correspondences.good.minAngleDifference1") * M_PI/180);
  float const maxOffsetDistance1 = config::lookup<double>("correspondences.good.maxOffsetDifference1");
  float const minIntersectionOverUnion = config::lookup<double>("correspondences.good.minIntersectionOverUnion");

  float const minDotProduct2 = std::cos(config::lookup<double>("correspondences.good.minAngleDifference2") * M_PI/180);
  float const maxOffsetDistance2 = config::lookup<double>("correspondences.good.maxOffsetDifference2");
  float const minIntersectionOverMin = config::lookup<double>("correspondences.good.minIntersectionOverMin");

  std::function<Hull(Hull const &, Eigen::Vector3d const &)> transformHull;
  if(config::lookup<bool>("correspondences.good.shiftHullByCentroid"))
  {
    transformHull = [](Hull const & hull, Eigen::Vector3d const & centroid)
    {
      auto transform_point = [](nrt::Point3D<float> const & p, Eigen::Vector3d const & c)
      { return nrt::Point3D<float>(p.x() - c.x(), p.y() - c.y(), p.z() - c.z()); };

      std::vector<nrt::Point3D<float>> hullTransformed(hull.size());
      std::transform(hull.begin(), hull.end(), hullTransformed.begin(),
          std::bind(transform_point, std::placeholders::_1, centroid));
      return hullTransformed;
    };
  }
  else
  {
    transformHull = [](Hull const & hull, Eigen::Vector3d const &)
    { return hull; };
  }

  // Find planes with similar hulls
  for(size_t i=0; i<frame1Planes.size(); ++i)
  {
    size_t bestIdx1 = std::numeric_limits<size_t>::max();
    float bestScore1 = std::numeric_limits<float>::infinity();

    size_t bestIdx2 = std::numeric_limits<size_t>::max();
    float bestScore2 = std::numeric_limits<float>::infinity();

    auto const & plane1 = frame1Planes[i];
    auto const & p1 = plane1.plane;
    //auto const & c1 = plane1.plane.projection(plane1.centroid->cast<float>());
    assert(plane1.centroid);

    for(size_t j=0; j<frame2Planes.size(); ++j)
    {
      auto const & plane2 = frame2Planes[j];
      auto const & p2 = plane2.plane;
      //auto const & c2 = plane2.plane.projection(plane2.centroid->cast<float>());
      assert(plane2.centroid);

      auto const averagePlane = polygonhelpers::findAveragePlane({plane1,plane2});

      const double normalDot  = p1.normal().dot(p2.normal());
      const double offsetDiff = std::abs(plane1.plane.offset() - plane2.plane.offset());
      //std::abs( (c1 - c2).dot(averagePlane.normal()) );

      if(normalDot < minDotProduct1) continue;

      if(offsetDiff < maxOffsetDistance1)
      {
        Hull const hull1Transformed = transformHull(plane1.hull, *plane1.centroid);
        Hull const hull2Transformed = transformHull(plane2.hull, *plane2.centroid);

        auto const projectedHull1 = polygonhelpers::projectPolygon(hull1Transformed, averagePlane);
        auto const projectedHull2 = polygonhelpers::projectPolygon(hull2Transformed, averagePlane);

        std::vector<polygonhelpers::Polygon> intersections;
        boost::geometry::intersection(projectedHull1, projectedHull2, intersections);

        if(!intersections.empty())
        {
          std::vector<polygonhelpers::Polygon> unions;
          boost::geometry::union_(projectedHull1, projectedHull2, unions);

          if(!unions.empty())
          {
            float const i_area = boost::geometry::area(polygonhelpers::findLargestPolygon(intersections));
            float const u_area = boost::geometry::area(polygonhelpers::findLargestPolygon(unions));
            float const intersection_over_union = i_area / u_area;

            if(intersection_over_union >= minIntersectionOverUnion)
            {
              float const score = offsetDiff * (1.0 - intersection_over_union);

              if(i_area > 0.0 && u_area > 0.0 && score < bestScore1)
              {
                bestScore1 = score;
                bestIdx1 = j;
              }
            }
          }
        }
      }

      // Case 2 computations
      if(offsetDiff >= maxOffsetDistance2 || normalDot < minDotProduct2) continue;

      auto const projectedHull1 = polygonhelpers::projectPolygon(plane1.hull, averagePlane);
      auto const projectedHull2 = polygonhelpers::projectPolygon(plane2.hull, averagePlane);
      std::vector<polygonhelpers::Polygon> intersections;
      boost::geometry::intersection(projectedHull1, projectedHull2, intersections);

      if(!intersections.empty())
      {
        float const i_area  = boost::geometry::area(polygonhelpers::findLargestPolygon(intersections));
        float const p1_area = boost::geometry::area(projectedHull1);
        float const p2_area = boost::geometry::area(projectedHull2);
        float const intersection_over_min = i_area / std::min(p1_area, p2_area);

        if(intersection_over_min >= minIntersectionOverMin)
        {
          float const score = offsetDiff * (1.0 - intersection_over_min);

          if(i_area > 0.0 && p1_area > 0.0 && p2_area > 0.0 && score < bestScore2)
          {
            bestScore2 = score;
            bestIdx2 = j;
          }
        }
      }
    } // end frame2 loop

    if(bestIdx1 < std::numeric_limits<size_t>::max())
      correspondences.emplace_back(i, bestIdx1, bestScore1);
    else if(bestIdx2 < std::numeric_limits<size_t>::max())
      correspondences.emplace_back(i, bestIdx2, bestScore2);
  } // end frame1 loop

  return correspondences;
}

// ######################################################################
nrt::Correspondences findPossibleCorrespondences(
    std::vector<DetectedPlane> const & frame1Planes, std::vector<DetectedPlane> const & frame2Planes,
    float const maxAngle = 20, float const maxRhoDifference = 0.25, float const areaThreshold = 0.5f)
{
  float const minDotProduct = cos(maxAngle*M_PI/180);

  nrt::Correspondences correspondences;

  for(size_t i=0; i<frame1Planes.size(); ++i)
  {
    DetectedPlane const & d1 = frame1Planes[i];
    Eigen::Hyperplane<float, 3> const & p1 = d1.plane;

    for(size_t j=0; j<frame2Planes.size(); ++j)
    {
      DetectedPlane const & d2 = frame2Planes[j];
      Eigen::Hyperplane<float, 3> const & p2 = d2.plane;

      float const d_offset = std::abs(p1.offset()-p2.offset());
      float const dotProduct = p1.normal().dot(p2.normal());
      assert(d1.area && d2.area);
      float const area = std::abs(*d1.area - *d2.area);

      if(d_offset < maxRhoDifference && dotProduct >= minDotProduct && area <= std::max(*d1.area, *d2.area)*areaThreshold)
        correspondences.emplace_back(i, j, d_offset * dotProduct);
    }
  }

  return correspondences;
}

namespace pathak
{
  const double ChiSq_1_00000001pc = 35.9999; ///< dof 1, sign. level 0.0000001%
  const double ChiSq_1_00001pc    = 23.9284; ///< dof 1, sign. level 0.0001%
  const double ChiSq_1_0001pc     = 19.5114; ///< dof 1, sign. level 0.001%
  const double ChiSq_1_001pc      = 15.1366; ///< dof 1, sign. level 0.01%
  const double ChiSq_1_01pc       = 10.82;   ///< dof 1, sign. level 0.1%
  const double ChiSq_1_05pc       = 7.8794;  ///< dof 1, sign. level 0.5%
  const double ChiSq_1_1pc        = 6.635;   ///< dof 1, sign. level 1%
  const double ChiSq_1_5pc        = 3.841;   ///< dof 1, sign. level 5%
  const double ChiSq_1_25pc       = 1.3233;  ///< dof 1, sign. level 25%

  const double ChiSq_2_00001pc = 27.6310; ///< dof 2, sign. level 0.0001%
  const double ChiSq_2_0001pc  = 23.0258; ///< dof 2, sign. level 0.001%
  const double ChiSq_2_01pc    = 13.8155; ///< dof 2, sign. level 0.1%
  const double ChiSq_2_1pc     = 9.210;   ///< dof 2, sign. level 1%
  const double ChiSq_2_5pc     = 5.991;   ///< dof 2, sign. level 5%
  const double ChiSq_2_25pc    = 2.7725;  ///< dof 2, sign. level 25%

  const double ChiSq_3_00001pc = 30.6649; ///< dof 3, sign. level 0.0001%
  const double ChiSq_3_0001pc  = 25.9017; ///< dof 3, sign. level 0.001%
  const double ChiSq_3_001pc   = 21.1075; ///< dof 3, sign. level 0.01%
  const double ChiSq_3_01pc    = 11.345;  ///< dof 3, sign. level 0.1%
  const double ChiSq_3_1pc     = 11.345;  ///< dof 3, sign. level 1%
  const double ChiSq_3_5pc     = 7.815;   ///< dof 3, sign. level 5%
  const double ChiSq_3_10pc    = 6.2513;  ///< dof 3, sign. level 10%
  const double ChiSq_3_25pc    = 4.1083;  ///< dof 3, sign. level 25%
  const double ChiSq_3_50pc    = 2.3659;  ///< dof 3, sign. level 50%
  const double ChiSq_3_75pc    = 1.2125;  ///< dof 3, sign. level 75%

  const double ChiSq_8_0001pc  = 37.3315; ///< dof 8, sign. level 0.001%
  const double ChiSq_14_0001pc = 48.7159; ///< dof 14, sign. level 0.001%

  enum class Parallelism {NotParallel = 0, Parallel = 1, AntiParallel = 2};

  // ######################################################################
  Parallelism findParallelism(DetectedPlane const & p1, DetectedPlane const & p2)
  {
    static float const threshold = std::cos(5*M_PI/180.0);
    float const dot = p1.plane.normal().dot(p2.plane.normal());
    bool const anti  = std::signbit(dot); // true if we might be anti-parallel

    if(std::abs(dot) >= threshold)
      return anti ? Parallelism::AntiParallel : Parallelism::Parallel;
    else
      return Parallelism::NotParallel;
  }

  // ######################################################################
  nrt::Optional<float> parallelConsistencyTest(DetectedPlane const & i1, DetectedPlane const & i2,
      DetectedPlane const & j1, DetectedPlane const & j2)
  {
    Parallelism const parallel1 = findParallelism(i1, j1);
    Parallelism const parallel2 = findParallelism(i2, j2);

    // Make sure they are both either parallel or antiparallel
    if(parallel1 == Parallelism::NotParallel || parallel2 == Parallelism::NotParallel)
      return nrt::OptionalEmpty;
    if(parallel1 != parallel2)
      return nrt::OptionalEmpty;

    double di = i1.plane.offset() - i2.plane.offset();
    double dj = j1.plane.offset() - j2.plane.offset();

    // If the planes are anti-parallel
    if(parallel1 == Parallelism::AntiParallel) dj = -dj;

    double const dmd = di - dj;
    double const ret = dmd*dmd / (i1.rhoCovariance + i2.rhoCovariance + j1.rhoCovariance + j2.rhoCovariance);

    if(ret <= ChiSq_1_01pc) return ret;

    return nrt::OptionalEmpty;
  }

  // ######################################################################
  nrt::Optional<float> crossAngleConsistencyTest(DetectedPlane const & i1, DetectedPlane const & i2,
      DetectedPlane const & j1, DetectedPlane const & j2)
  {
    float const dot1 = j1.plane.normal().dot(i1.plane.normal());
    float const dotVariance1 =
      j1.plane.normal().dot(i1.normalCovariance.cast<float>() * j1.plane.normal()) +
      i1.plane.normal().dot(j1.normalCovariance.cast<float>() * i1.plane.normal());

    float const dot2 = j2.plane.normal().dot(i2.plane.normal());
    float const dotVariance2 =
      j2.plane.normal().dot(i2.normalCovariance.cast<float>() * j2.plane.normal()) +
      i2.plane.normal().dot(j2.normalCovariance.cast<float>() * i2.plane.normal());

    float const dmd = dot1 - dot2;
    float const ret = (dmd*dmd) / (dotVariance1 + dotVariance2);

    if(ret <= ChiSq_1_00001pc) return ret;
    //if(ret <= ChiSq_1_001pc) return ret;

    return nrt::OptionalEmpty;
  }

  // ######################################################################
  nrt::Optional<float> translationConsistencyTest(DetectedPlane const & i1, DetectedPlane const & i2, Eigen::Translation3f const & translation,
      Eigen::Matrix3d const & translationCovariance)
  {
    Eigen::Vector3f const & t = translation.vector();
    float const num = i1.plane.normal().dot(t) - (i1.plane.offset() - i2.plane.offset());
    float const ret = (num * num) / (i1.rhoCovariance + i2.rhoCovariance +
        (t.transpose() * i1.normalCovariance.cast<float>() * t) +
        (i1.plane.normal().transpose() * translationCovariance.cast<float>() * i1.plane.normal()));

    //if(ret <= ChiSq_1_5pc)
    if(ret <= ChiSq_1_00001pc)
      return ret;
    else
      return nrt::OptionalEmpty;
  }


} // end namespace pathak

// ######################################################################
nrt::Optional<float> parallelConsistencyTest(DetectedPlane const & i1, DetectedPlane const & i2,
    DetectedPlane const & j1, DetectedPlane const & j2, float const minAngle = 0.1f)
{
  // Ensure that the matches are near parallel
  float const normalThreshold = std::cos(minAngle);
  if(i1.plane.normal().dot(i2.plane.normal()) >= normalThreshold && j1.plane.normal().dot(j2.plane.normal()) >= normalThreshold)
  {
    float const delta = (i1.plane.offset() - i2.plane.offset()) - (j1.plane.offset() - j2.plane.offset());
    return std::abs(delta);
  }

  return nrt::OptionalEmpty;
}

// ######################################################################
nrt::Optional<float> crossAngleConsistencyTest(DetectedPlane const & i1, DetectedPlane const & i2,
    DetectedPlane const & j1, DetectedPlane const & j2)
{
  float const v1 = j1.plane.normal().dot(i1.plane.normal());
  float const v2 = j2.plane.normal().dot(i2.plane.normal());

  return std::abs(v1 - v2);
}

// ######################################################################
// Makes a set of correspondences unique by taking the best (lowest) scoring
// result for both the source and target indices
nrt::Correspondences findUniqueCorrespondences(nrt::Correspondences correspondences)
{
  std::sort(correspondences.begin(), correspondences.end(), [](nrt::Correspondence const & c1, nrt::Correspondence const & c2) { return c1.score < c2.score; });
  nrt::Correspondences uniqueCorrespondences;
  for(nrt::Correspondence const & c : correspondences)
  {
    if(std::find_if(uniqueCorrespondences.begin(), uniqueCorrespondences.end(),
          [c](nrt::Correspondence const & d) { return c.sourceIndex == d.sourceIndex; }) != uniqueCorrespondences.end())
      continue;

    if(std::find_if(uniqueCorrespondences.begin(), uniqueCorrespondences.end(),
          [c](nrt::Correspondence const & d) { return c.targetIndex == d.targetIndex; }) != uniqueCorrespondences.end())
      continue;

    uniqueCorrespondences.push_back(c);
  }
  return uniqueCorrespondences;
}


// ######################################################################
nrt::Correspondences findMaxParallelSet(nrt::Correspondence const & currentPair,
    nrt::Correspondences const & translationConsistentSet, nrt::Correspondences const & c1)
{
  nrt::Correspondences relevantSet;
  for(nrt::Correspondence const & rc : translationConsistentSet)
  {
    nrt::Correspondence const & pair1 = c1[rc.sourceIndex];
    nrt::Correspondence const & pair2 = c1[rc.targetIndex];

    if(currentPair == pair1)
      relevantSet.push_back(pair2);
    else if(currentPair == pair2)
      relevantSet.push_back(pair1);
  }

  return findUniqueCorrespondences(relevantSet);
}

// ######################################################################
nrt::Correspondences findMaxRotationSet(nrt::Correspondence const & currentPair,
    nrt::Correspondences const & rotationConsistentSet, nrt::Correspondences const & c1,
    std::vector<DetectedPlane> const & frame1Planes, std::vector<DetectedPlane> const & frame2Planes,
    float const rotationThreshold = 0.90)
{
  //// Line 5 from Algorithm 2 MUMC
  // all level 1 (plane to plane) correspondences that are in rotationConsistentSet with the currentPair
  nrt::Correspondences currentConsistentSet; currentConsistentSet.reserve(rotationConsistentSet.size());

  // Use tuples that have the current pair in them
  for(nrt::Correspondence const & rc : rotationConsistentSet)
  {
    nrt::Correspondence const & pair1 = c1[rc.sourceIndex];
    nrt::Correspondence const & pair2 = c1[rc.targetIndex];

    if(currentPair == pair1)
      currentConsistentSet.push_back(pair2);
    else if(currentPair == pair2)
      currentConsistentSet.push_back(pair1);
  }

  DetectedPlane const & currentPlane1 = frame1Planes[currentPair.sourceIndex];
  DetectedPlane const & currentPlane2 = frame2Planes[currentPair.targetIndex];

  // remove dups
  std::sort(currentConsistentSet.begin(), currentConsistentSet.end());
  currentConsistentSet.erase(std::unique(currentConsistentSet.begin(), currentConsistentSet.end()), currentConsistentSet.end());

  nrt::Correspondences largestSet; // gamma non_parallell in paper
  float setScore = std::numeric_limits<double>::lowest();

  // Find the maximal consistent set
  //  Pick another correspondence, compute R, and then find all the others that fit with it
  for(size_t i = 0, end = currentConsistentSet.size(); i != end; ++i)
  {
    nrt::Correspondence outerPair = currentConsistentSet[i];

    DetectedPlane const & outerPlane1 = frame1Planes[outerPair.sourceIndex];
    DetectedPlane const & outerPlane2 = frame2Planes[outerPair.targetIndex];

    // Compute R
    Eigen::Matrix3f const R = findRotation({currentPair, outerPair}, frame1Planes, frame2Planes);

    float const currentScore = currentPlane1.plane.normal().dot(R * currentPlane2.plane.normal());
    float const outerScore   = outerPlane1.plane.normal().dot(R * outerPlane2.plane.normal());

    // Make sure the computed rotation is valid
    // TODO: Change this to the consistentWithRotation test from Pathak
    if(currentScore < rotationThreshold || outerScore < rotationThreshold)
      continue;

    // Create a consistent set
    nrt::Correspondences currentSet;
    float currentSetScore = 0;
    for(size_t j = 0; j != end; ++j)
    {
      if(i == j) continue;

      nrt::Correspondence const & currentCorr = currentConsistentSet[j];
      DetectedPlane const & p1 = frame1Planes[currentCorr.sourceIndex];
      DetectedPlane const & p2 = frame2Planes[currentCorr.targetIndex];

      const float score = p1.plane.normal().dot(R*p2.plane.normal());

      if(score >= rotationThreshold)
      {
        currentSet.push_back(currentConsistentSet[j]);
        currentSetScore += score;
      }
    } // end creating local set loop

    const size_t cSize = currentSet.size();
    const size_t lSize = largestSet.size();

    if(cSize > lSize)
    {
      setScore = currentSetScore;
      largestSet = std::move(currentSet);
    }
    else if(cSize == lSize && currentSetScore > setScore)
    {
      setScore = currentSetScore;
      largestSet = std::move(currentSet);
    }
  } // end maximal consistent set loop

  // Line 7 from Algorithm 2 MUMC
  nrt::Correspondences resultSet;
  setScore = std::numeric_limits<double>::infinity();

  auto constLargestSet = largestSet.const_begin();

  for(size_t i = 0; i < largestSet.size(); ++i)
  {
    // Pick second pair
    nrt::Correspondence const & secondCorr = constLargestSet[i];

    for(size_t j = i+1; j < largestSet.size(); ++j)
    {
      // Pick third pair
      nrt::Correspondence const & thirdCorr = constLargestSet[j];

      // Enforce that i and j are not parallel or anti-parallel to each other (by construction they will not be parallel
      // or anti-parallel to currentPair)
      auto parallelOne = pathak::findParallelism(frame1Planes[secondCorr.sourceIndex], frame1Planes[thirdCorr.sourceIndex]);
      auto parallelTwo = pathak::findParallelism(frame2Planes[secondCorr.targetIndex], frame2Planes[thirdCorr.targetIndex]);

      if(parallelOne != pathak::Parallelism::NotParallel || parallelTwo != pathak::Parallelism::NotParallel)
        continue;

      // Compute the translation
      double translationUncertainty;
      Eigen::Matrix3d translationCovariance;
      Eigen::Translation3f const translation = pathak::findTranslation(
          {currentPair, secondCorr, thirdCorr}, frame1Planes, frame2Planes,
          /*rotationEstimate*/ nrt::OptionalEmpty, /*rank*/ nrt::OptionalEmpty, translationCovariance,
          translationUncertainty);
      if(translationUncertainty == std::numeric_limits<double>::infinity()) continue;


      // Find biggest consistent set
      nrt::Correspondences currentSet;
      float currentScore = 0;
      for(size_t k = 0; k < largestSet.size(); ++k)
      {
        nrt::Correspondence const & fourthCorr = constLargestSet[k];

        auto score = pathak::translationConsistencyTest(frame1Planes[fourthCorr.sourceIndex], frame2Planes[fourthCorr.targetIndex],
                                                translation, translationCovariance);
        if(score)
        {
          currentSet.push_back(fourthCorr);
          currentScore += *score;
        }
      }

      if(currentSet.size() > resultSet.size() ||
          (currentSet.size() == resultSet.size() && currentScore < setScore))
      {
        setScore = currentScore;
        resultSet = std::move(currentSet);
      }
    }
  }

  return findUniqueCorrespondences(resultSet);
}

// ######################################################################
Eigen::Isometry3d findTransformMUMCFast (
    std::vector<DetectedPlane> const & frame1Planes, std::vector<DetectedPlane> const & frame2Planes,
    nrt::Optional<nrt::Correspondences&> correspondencesOutput, nrt::Optional<int&> translationRankOutput,
    nrt::Optional<Eigen::Matrix3d&> translationCovarianceOutput, nrt::Optional<Eigen::Matrix3d&> rotationCovarianceOutput)
{
  nrt::Correspondences c1 = findPossibleCorrespondences(frame1Planes, frame2Planes, 
      config::lookup<double>("correspondences.mumcfast.possible.angle"),
      config::lookup<double>("correspondences.mumcfast.possible.offset"),
      config::lookup<double>("correspondences.mumcfast.possible.area"));

      ///*angle*/10, /*offset*/0.25, /*area*/1000.5);
  if(c1.size() < 2)
  {
    std::cerr << "    Empty dempty" << std::endl;
    return Eigen::Isometry3d::Identity();
  }

  // Algorithm 1 of MUMC
  nrt::Correspondences translationConsistentSet;
  nrt::Correspondences rotationConsistentSet;
  for(size_t k=0; k<c1.size()-1; ++k)
  {
    nrt::Correspondence const & p_k = c1[k];
    DetectedPlane const & k1 = frame1Planes[p_k.sourceIndex];
    DetectedPlane const & k2 = frame2Planes[p_k.targetIndex];

    for(size_t i=k+1; i<c1.size(); ++i)
    {
      nrt::Correspondence const & p_i = c1[i];
      DetectedPlane const & i1 = frame1Planes[p_i.sourceIndex];
      DetectedPlane const & i2 = frame2Planes[p_i.targetIndex];

      nrt::Optional<float> crossAngleConsistency = pathak::crossAngleConsistencyTest(k1,k2,i1,i2);
      if(!crossAngleConsistency) continue;

      nrt::Optional<float> parallelConsistency = pathak::parallelConsistencyTest(k1,k2,i1,i2);

      if(parallelConsistency)
        translationConsistentSet.emplace_back(k, i, *parallelConsistency);
      else
        rotationConsistentSet.emplace_back(k, i, *crossAngleConsistency);
    }
  }

  nrt::Correspondences K;
  for(nrt::Correspondence const & c : translationConsistentSet)
  {
    K.push_back(c1[c.sourceIndex]);
    K.push_back(c1[c.targetIndex]);
  }
  for(nrt::Correspondence const & c : rotationConsistentSet)
  {
    K.push_back(c1[c.sourceIndex]);
    K.push_back(c1[c.targetIndex]);
  }
  std::sort(K.begin(), K.end());
  K.erase(std::unique(K.begin(), K.end()), K.end());

  nrt::Correspondences goodCorrespondences;
  for(nrt::Correspondence k : K)
  {
    DetectedPlane const & p1 = frame1Planes[k.sourceIndex];
    DetectedPlane const & p2 = frame2Planes[k.targetIndex];

    if(p1.plane.normal().dot(p2.plane.normal()) < .8) continue;

    k.score = (*p1.centroid - *p2.centroid).squaredNorm();

    if(k.score > 1.0) continue;

    goodCorrespondences.push_back(k);
  }

  goodCorrespondences = findUniqueCorrespondences(goodCorrespondences);



  double translationUncertainty;
  int translationRank;
  Eigen::Matrix3d translationCovariance;
  Eigen::Translation3f translation = pathak::findTranslation(goodCorrespondences, frame1Planes, frame2Planes, nrt::OptionalEmpty, translationRank, translationCovariance, translationUncertainty);
  double rotationUncertainty;
  Eigen::Matrix3d rotationCovariance;
  Eigen::Matrix3f rotation = pathak::findRotation(goodCorrespondences, frame1Planes, frame2Planes, rotationCovariance, rotationUncertainty);

  auto transform = Eigen::Isometry3d::Identity();
  transform.rotate(rotation.cast<double>());
  transform.translation() = translation.vector().cast<double>();

  if(translationCovarianceOutput) *translationCovarianceOutput = translationCovariance;
  if(rotationCovarianceOutput) *rotationCovarianceOutput = rotationCovariance;

  if(correspondencesOutput) *correspondencesOutput = goodCorrespondences;
  if(translationRankOutput) *translationRankOutput = translationRank;
  return transform;
}

// ######################################################################
Eigen::Isometry3d findTransformMUMC (
    std::vector<DetectedPlane> const & frame1Planes, std::vector<DetectedPlane> const & frame2Planes,
    nrt::Optional<nrt::Correspondences&> correspondencesOutput, nrt::Optional<int&> translationRank, nrt::Optional<nrt::Correspondences&> Koutput)
{
  frame_timing::guard mumc_timing("MUMC");

  frame_timing::start("Algorithm 1");
  nrt::Correspondences c1 = findPossibleCorrespondences(frame1Planes, frame2Planes, /*angle*/10, /*offset*/0.5, /*area*/1000.5);
  if(c1.size() < 2)
  {
    std::cerr << "    Empty dempty" << std::endl;
    frame_timing::stop("Algorithm 1");
    return Eigen::Isometry3d::Identity();
  }

  // Algorithm 1 of MUMC
  nrt::Correspondences translationConsistentSet;
  nrt::Correspondences rotationConsistentSet;
  for(size_t k=0; k<c1.size()-1; ++k)
  {
    nrt::Correspondence const & p_k = c1[k];
    DetectedPlane const & k1 = frame1Planes[p_k.sourceIndex];
    DetectedPlane const & k2 = frame2Planes[p_k.targetIndex];

    for(size_t i=k+1; i<c1.size(); ++i)
    {
      nrt::Correspondence const & p_i = c1[i];
      DetectedPlane const & i1 = frame1Planes[p_i.sourceIndex];
      DetectedPlane const & i2 = frame2Planes[p_i.targetIndex];

      nrt::Optional<float> crossAngleConsistency = pathak::crossAngleConsistencyTest(k1,k2,i1,i2);
      if(!crossAngleConsistency) continue;

      nrt::Optional<float> parallelConsistency = pathak::parallelConsistencyTest(k1,k2,i1,i2);

      if(parallelConsistency)
        translationConsistentSet.emplace_back(k, i, *parallelConsistency);
      else
        rotationConsistentSet.emplace_back(k, i, *crossAngleConsistency);
    }
  }
  frame_timing::stop("Algorithm 1");

  frame_timing::start("Algorithm 2");
  // Algorithm 2 of MUMC
  nrt::Correspondences K;
  for(nrt::Correspondence const & c : translationConsistentSet)
  {
    K.push_back(c1[c.sourceIndex]);
    K.push_back(c1[c.targetIndex]);
  }
  for(nrt::Correspondence const & c : rotationConsistentSet)
  {
    K.push_back(c1[c.sourceIndex]);
    K.push_back(c1[c.targetIndex]);
  }
  std::sort(K.begin(), K.end());
  K.erase(std::unique(K.begin(), K.end()), K.end());

  if(Koutput)
  {
    *Koutput = K;
  }

  nrt::Correspondences bestCorrespondences;
  double bestScore = std::numeric_limits<double>::infinity();
  Eigen::Isometry3d bestTransform = Eigen::Isometry3d::Identity();
  size_t bestTranslationRank = 0;
  for(nrt::Correspondence const & k : K)
  {
    frame_timing::start("Max Sets");
    // Line 5+7
    nrt::Correspondences maxRotationSet = findMaxRotationSet(k, rotationConsistentSet, c1, frame1Planes, frame2Planes, 0.8);

    // Line 6
    nrt::Correspondences maxParallelSet = findMaxParallelSet(k, translationConsistentSet, c1);
    frame_timing::stop("Max Sets");


    if(maxRotationSet.size() + maxParallelSet.size() < 4) continue;

    // Line 8
    nrt::Correspondences gamma;
    gamma.insert(gamma.end(), maxRotationSet.begin(), maxRotationSet.end());
    gamma.insert(gamma.end(), maxParallelSet.begin(), maxParallelSet.end());

    // Line 9
    frame_timing::start("Find Transform");
    double translationUncertainty;
    int translationRank;
    Eigen::Translation3f translation =
      pathak::findTranslation(gamma, frame1Planes, frame2Planes, nrt::OptionalEmpty, translationRank, nrt::OptionalEmpty, translationUncertainty);
    double rotationUncertainty;
    Eigen::Matrix3f rotation = pathak::findRotation(gamma, frame1Planes, frame2Planes, nrt::OptionalEmpty, rotationUncertainty);
    frame_timing::stop("Find Transform");


    double const uncertaintyVolume = translationUncertainty * rotationUncertainty;


    if(translation.vector().norm() > 0.25) continue;


    if(uncertaintyVolume < bestScore)
    {
      bestCorrespondences = std::move(gamma);
      bestScore = uncertaintyVolume;
      bestTransform = Eigen::Isometry3d::Identity();
      bestTransform.rotate(rotation.cast<double>());
      bestTransform.translation() = translation.vector().cast<double>();
      bestTranslationRank = translationRank;
    }
  }
  frame_timing::stop("Algorithm 2");

  if(correspondencesOutput)
    *correspondencesOutput = bestCorrespondences;
  if(translationRank)
    *translationRank = bestTranslationRank;

  return bestTransform;
}

