#pragma once

#include <nrt/PointCloud2/PointCloud2.H>
#include <vector>

typedef size_t groupid_t;

struct Distance { float value; };

// ######################################################################
//! Get the row number of a given velodyne point index
/*! Assumes data is in column major order in the point cloud */
template<size_t numRows> inline size_t velodyneIdx2Row(size_t idx);

// Velodyne HDL-32E row to sequence (laser) mapping
// Row Seq
// 0	0
// 1	2
// 2	4
// 3	6
// 4	8
// 5	10
// 6	12
// 7	14
// 8	16
// 9	18
// 10	20
// 11	22
// 12	24
// 13	26
// 14	28
// 15	30
// 16	1
// 17	3
// 18	5
// 19	7
// 20	9
// 21	11
// 22	13
// 23	15
// 24	17
// 25	19
// 26	21
// 27	23
// 28	25
// 29	27
// 30	29
// 31	31
template<> inline size_t velodyneIdx2Row<32>(size_t idx)
{
  size_t const seq = idx % 32;
  size_t const row = (seq % 2) ? 16+seq/2 : seq/2;
  return row;
}

// Designed around our loaded Karlsruhe dataset, which we load
// in column major order but keep the lasers in the proper order
template<> inline size_t velodyneIdx2Row<64>(size_t idx)
{
  return idx % 64;
}

// ######################################################################
//! Get the index of the first point of a given row
/*! Maps from row to sequence number */
template<size_t numRows> inline size_t getVelodyneFirstRowIdx(size_t row);

template<> inline size_t getVelodyneFirstRowIdx<32>(size_t row)
{ return (row < 16) ? (row*2) : ((row-16)*2+1); }

// Using karlsruhe dataset
template<> inline size_t getVelodyneFirstRowIdx<64>(size_t row)
{ return row; }

// ######################################################################
//! Get the group ID given a row, and a sequence number
template<size_t numRows> inline groupid_t groupId(size_t const row, size_t const seq);

template<> inline groupid_t groupId<32>(size_t const row, size_t const seq)
{ return row + 32ul * seq; }

template<> inline groupid_t groupId<64>(size_t const row, size_t const seq)
{ return row + 64ul * seq; }

// ######################################################################
//! Extract the row from a given group id
template<size_t numRows> inline size_t rowFromGroupId(groupid_t const groupid);

template<> inline size_t rowFromGroupId<32>(groupid_t const groupid)
{ return groupid % 32; }

template<> inline size_t rowFromGroupId<64>(groupid_t const groupid)
{ return groupid % 64; }

// ######################################################################
//! Extract the sequence from a given group id
template<size_t numRows> inline size_t seqFromGroupId(groupid_t const groupid);

template<> inline size_t seqFromGroupId<32>(groupid_t const groupid)
{ return (groupid - rowFromGroupId<32>(groupid)) / 32; }

template<> inline size_t seqFromGroupId<64>(groupid_t const groupid)
{ return (groupid - rowFromGroupId<64>(groupid)) / 64; }

// ######################################################################
//! Get a reference to a group from an id
template <size_t numRows, class T>
inline T & groupFromId(std::array<std::vector<T>, numRows> & groups, groupid_t const gid)
{ return groups[rowFromGroupId<numRows>(gid)][seqFromGroupId<numRows>(gid)]; }

// ######################################################################
//! Get a reference to a group from an id (const version)
template <size_t numRows, class T>
inline T const & groupFromId(std::array<std::vector<T>, numRows> const & groups, groupid_t const gid)
{ return groups[rowFromGroupId<numRows>(gid)][seqFromGroupId<numRows>(gid)]; }

// ######################################################################
//! Given a row, find the offset to add to the current index to go up a row
template<size_t numRows> inline int getPositiveRowIncrement(size_t const row);

// see above sequence/row map for hdl32e
template<> inline int getPositiveRowIncrement<32>(size_t const row)
{
  static constexpr std::array<int, 32> positiveRowIncrement = {{
    +2, +2, +2, +2, +2, +2, +2, +2, +2, +2, +2, +2, +2, +2, +2, -29,
      +2, +2, +2, +2, +2, +2, +2, +2, +2, +2, +2, +2, +2, +2, +2,
      std::numeric_limits<int>::max()
  }};
  return positiveRowIncrement[row];
}

// karlsruhe is loaded with sequences in order
template<> inline int getPositiveRowIncrement<64>(size_t const row)
{
  return 1;
}

// ######################################################################
//! Given a row, find the offset to add to the current index to go down a row
template<size_t numRows> inline int getNegativeRowIncrement(size_t const row);

// see above sequence/row map for hdl32e
template<> inline int getNegativeRowIncrement<32>(size_t const row)
{
  static constexpr std::array<int, 32> negativeRowIncrement = {{
    std::numeric_limits<int>::max(),
      -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2,
      +29, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2,
  }};
  return negativeRowIncrement[row];
}

// karlsruhe is loaded with sequences in order
template<> inline int getNegativeRowIncrement<64>(size_t const row)
{
  return -1;
}

//! Computes the shortest distance between two line segments S1, defined by s1_begin and s1_end,
//! and S2, defined by s2_begin and s2_end
/*! see: http://geomalgorithms.com/a07-_distance.html
    see: http://geomalgorithms.com/a07-_distance.html#dist3D_Segment_to_Segment() */
float shortestSquaredDistanceBetweenSegments(Eigen::Vector3f const & s1_begin,
                                             Eigen::Vector3f const & s1_end,
                                             Eigen::Vector3f const & s2_begin,
                                             Eigen::Vector3f const & s2_end);

// Removes a set of indices from a vector
template <class T>
inline void removeIndices(std::vector<T> & v, std::vector<typename std::vector<T>::iterator> const & indices)
{
  auto v_last = v.rbegin();

  for(auto indit = indices.rbegin(); indit != indices.rend(); ++indit)
  {
    *(*indit) = *v_last;
    ++v_last;
  }

  v.erase(v.end()-indices.size(), v.end());
}
