#include "FrameTiming.H"
#include <boost/variant.hpp>
#include <vector>
#include <memory>
#include <unordered_map>
#include <algorithm>
#include <iostream>
#include <boost/format.hpp>
#include <iomanip>
#include <sstream>

namespace frame_timing
{

  struct TaskInfo : std::enable_shared_from_this<TaskInfo>
  {
    TaskInfo(std::shared_ptr<TaskInfo> parent, std::string const & name) :
       parent(parent), name(name)
    { }

    std::shared_ptr<TaskInfo> get_subtask(std::string const & name)
    {
      auto task = std::find_if(subtasks.begin(), subtasks.end(),
          [&name](std::shared_ptr<TaskInfo> t) { return t->name == name; });

      if(task == subtasks.end())
      {
        auto new_task = std::make_shared<TaskInfo>(shared_from_this(), name);
        subtasks.push_back(new_task);
        return new_task;
      }
      else
      {
        return *task;
      }
    }

    std::weak_ptr<TaskInfo> parent;
    std::vector<std::shared_ptr<TaskInfo>> subtasks;

    std::string const name;
    Clock::time_point start;
    Duration total;
  };

  struct FrameInfo
  {
    FrameInfo() :
      root_task(std::make_shared<TaskInfo>(nullptr, "frame")),
      current_task(root_task)
    { 
      root_task->parent = root_task;
    }

    FrameInfo(FrameInfo const &) = delete;

    std::shared_ptr<TaskInfo> root_task;
    std::shared_ptr<TaskInfo> current_task;
    int n_frames;
  };

  FrameInfo & get_frame() 
  { 
    static FrameInfo info;
    return info; 
  }
}

void frame_timing::begin_frame()
{
  auto & frame = get_frame();
  frame.current_task = frame.root_task;
  frame.root_task->start = Clock::now();
}

void frame_timing::start(std::string const & name)
{
  auto & frame = get_frame();
  auto task = frame.current_task->get_subtask(name);
  frame.current_task = task;

  task->start = Clock::now();
}

void frame_timing::stop(std::string const & name)
{
  auto now = Clock::now();

  auto & frame = get_frame();
  auto task = frame.current_task;
  if(name != task->name)
    throw std::runtime_error("Error calling frame_timing::stop(\"" + name + "\"), but current task is " + task->name);

  assert(task != frame.root_task);

  auto delta = now - task->start;
  task->total += delta;

  frame.current_task = std::shared_ptr<TaskInfo>(task->parent);
}

void frame_timing::end_frame()
{
  auto now = Clock::now();
  auto & frame = get_frame();

  frame.root_task->total += now - frame.root_task->start;
  ++frame.n_frames;
}

void frame_timing::clear()
{
  std::function<void(std::shared_ptr<frame_timing::TaskInfo>)> clear_impl = 
    [&clear_impl](std::shared_ptr<frame_timing::TaskInfo> task)
    {
      task->total = frame_timing::Duration(0);
      for(auto t : task->subtasks) clear_impl(t);
    };

  auto & frame = get_frame();
  frame.n_frames = 0;
  clear_impl(frame.root_task);
}

std::string frame_timing::frame_json()
{
  auto & frame = get_frame();

  std::stringstream ret;

  std::function<void(std::shared_ptr<TaskInfo>)> report_impl =
    [&report_impl, &frame, &ret](std::shared_ptr<TaskInfo> task)
    {

      ret << "{ \"name\": \"" << task->name << "\",\n";
      ret << "  \"time\": \"" << (task->total.count() / float(frame.n_frames)) << "\",\n";
      ret << "  \"tasks\": [\n";
      for(size_t i=0; i<task->subtasks.size(); ++i)
      {
        if(i != 0) ret << ",\n";
        report_impl(task->subtasks[i]);
      }
      ret << "  ]\n";
      ret << "}";
    };

  for(size_t i=0; i<frame.root_task->subtasks.size(); ++i)
  {
    if(i != 0) ret << ",\n";
    report_impl(frame.root_task->subtasks[i]);
  }

  return ret.str();
}


void frame_timing::internal::do_report()
{
  auto & frame = get_frame();

  size_t const spaces_per_level = 1;

  size_t max_name_length = 0;
  std::function<void(std::shared_ptr<TaskInfo>, size_t)> get_max_name_length =
    [&get_max_name_length, &max_name_length](std::shared_ptr<TaskInfo> task, size_t depth)
    {
      for(auto t : task->subtasks)
      {
        max_name_length = std::max(max_name_length + depth*spaces_per_level, t->name.size());
        get_max_name_length(t, depth+1);
      }
    };
  get_max_name_length(frame.root_task, 0);

  std::function<void(std::shared_ptr<TaskInfo>, size_t)> report_impl =
    [&report_impl, &frame, max_name_length](std::shared_ptr<TaskInfo> task, size_t indentation)
    {
      std::cout << "├";
      for(size_t i=0; i<indentation; ++i) 
        for(size_t j=0; j<spaces_per_level; ++j)
        std::cout << "─";

      std::cout << "[" << task->name << "]";
      int const extra_spaces = int(max_name_length+3) - int(task->name.size()) - int(indentation*spaces_per_level);
      for(int i=0; i<extra_spaces; ++i) std::cout << " ";


      std::cout << std::setprecision(2) << std::setw(8) << std::fixed << (task->total.count() / float(frame.n_frames)) << " ms" << std::endl;

      for(auto t : task->subtasks) report_impl(t, indentation+1);
    };

  std::cout << "------------------------- Timing -------------------------" << std::endl;
  for(auto t : frame.root_task->subtasks)
    report_impl(t, 0);

  auto frame_ms = (frame.root_task->total.count() / float(frame.n_frames));
  std::cout << std::endl << "Frame Time: " << frame_ms << " ms (" << 1.0/(frame_ms/1000.0) << " Hz)" << std::endl;
}
