#include "VelodyneNormals.H"
#include "VelodyneHelpers.H"
#include <nrt/PointCloud2/Common/Plane.H>
#include <nrt/PointCloud2/Features/Normals.H>
#include <array>

namespace
{
  // A table that converts between a row number, and a velodyne offset number
  static constexpr std::array<int, 32> rowTableArray =
  {{ velodyne32::rowTable(  0 ),
     velodyne32::rowTable(  1 ),
     velodyne32::rowTable(  2 ),
     velodyne32::rowTable(  3 ),
     velodyne32::rowTable(  4 ),
     velodyne32::rowTable(  5 ),
     velodyne32::rowTable(  6 ),
     velodyne32::rowTable(  7 ),
     velodyne32::rowTable(  8 ),
     velodyne32::rowTable(  9 ),
     velodyne32::rowTable( 10 ),
     velodyne32::rowTable( 11 ),
     velodyne32::rowTable( 12 ),
     velodyne32::rowTable( 13 ),
     velodyne32::rowTable( 14 ),
     velodyne32::rowTable( 15 ),
     velodyne32::rowTable( 16 ),
     velodyne32::rowTable( 17 ),
     velodyne32::rowTable( 18 ),
     velodyne32::rowTable( 19 ),
     velodyne32::rowTable( 20 ),
     velodyne32::rowTable( 21 ),
     velodyne32::rowTable( 22 ),
     velodyne32::rowTable( 23 ),
     velodyne32::rowTable( 24 ),
     velodyne32::rowTable( 25 ),
     velodyne32::rowTable( 26 ),
     velodyne32::rowTable( 27 ),
     velodyne32::rowTable( 28 ),
     velodyne32::rowTable( 29 ),
     velodyne32::rowTable( 30 ),
     velodyne32::rowTable( 31 ) }};

  // ######################################################################
  nrt::Indices getNeighbors(nrt::PointCloud2 const cloud, int idx, float distance, std::array<int, 32> const & rowtable)
  {
    int const row = velodyne32::idx2row(idx);
    int const col = velodyne32::idx2col(idx);

    double const rad_per_row = 1.34*M_PI/180.0;
    double const rad_per_col = 0.00287296996;
    int const cloudsize = cloud.size();
    int num_cols = cloudsize / 32;

    float pdist = cloud[idx].getVectorMap().norm();

    // Find the row/col bounds of the search
    int const nv = ceil(atan(distance / pdist) / rad_per_row / 2.0);
    int const nh = ceil(atan(distance / pdist) / rad_per_col / 2.0);
    int const rowmin = std::max(0,row-nv);
    int const rowmax = std::min(31,row+nv);
    int const colmin = col-nh;
    int const colmax = col+nh;

    // Cache a table of the row offsets
    static std::array<int, 32> localrowtable;
    for(int irow=rowmin, irow_end=rowmax; irow<=irow_end; ++irow)
      localrowtable[irow-rowmin] = rowtable[irow];
    int const * const rowend = &localrowtable[0] + (rowmax-rowmin)+1;

    nrt::Indices indices((colmax-colmin+1)*(rowmax-rowmin+1)+1);
    long unsigned int * indit = &indices[0];
    *indit = idx;
    ++indit;
    for(int icol=colmin, icol_end=colmax; icol<=icol_end; ++icol)
    {
      int *rowit = &localrowtable[0];
      int pcol = (icol < 0) ? (icol+num_cols)*32 : ((icol > num_cols-1) ? (icol - num_cols)*32 : icol*32);
      while(rowit != rowend)
      {
        *indit = pcol + *rowit;
        ++rowit;
        ++indit;
      }
    }

    return indices;
  }
}

// ######################################################################
nrt::PointCloud2 computeVelodyneNormals( nrt::PointCloud2 const input, float const distance, bool twoPass)
{
  return nrt::computePointCloudFeatures( input, VelodyneNormals( distance, twoPass ) );
}

// ######################################################################
VelodyneNormals::VelodyneNormals( double distance, bool twoPass,
                       nrt::PointCloud2::Geometry const & viewpoint ) :
  itsDistance( distance ),
  itsUseTwoPassCovariance( twoPass ),
  itsViewPoint( viewpoint )
{ }

VelodyneNormals::~VelodyneNormals()
{ }

nrt::PointCloud2 VelodyneNormals::computeFeatures( nrt::PointCloud2 const cloud )
{
  nrt::PointCloud2 output = cloud;
  output.addField<nrt::PointNormal>();

  computeFeaturesImpl( cloud, output.begin<nrt::PointNormal>(), output.end<nrt::PointNormal>() );

  return output;
}

nrt::PointCloud2 VelodyneNormals::computeFeatures( nrt::PointCloud2 const cloud, nrt::Indices const subset )
{
  nrt::PointCloud2 output = cloud;
  output.addField<nrt::PointNormal>();

  computeFeaturesImpl( cloud, output.subset_begin<nrt::PointNormal>( subset ), output.subset_end<nrt::PointNormal>( subset ) );

  return output;
}

nrt::Indices VelodyneNormals::velodynePatch( nrt::PointCloud2 const cloud, int const idx, int const dr, int const dc)
{
  int const row = velodyne32::idx2row(idx);
  int const col = velodyne32::idx2col(idx);

  int const ncols = cloud.size() / 32;

  int const minrow = std::max(0,  row - dr);
  int const maxrow = std::min(31, row + dr);

  int const mincol = std::max(0, col - dc);
  int const maxcol = std::min(ncols, col + dc);

  size_t nindices = 0;
  nrt::Indices indices((maxrow-minrow)*(maxcol-mincol));
  auto indices_it = indices.begin();
  for(int c=mincol; c<maxcol; ++c)
    for(int r=minrow; r<maxrow; ++r)
    {
      auto idx = velodyne32::index(r,c);
      if(cloud[idx].isValid())
      {
        *indices_it++ = velodyne32::index(r,c);
        ++nindices;
      }
    }
  indices.resize(nindices);

  return indices;
}

nrt::Indices VelodyneNormals::velodyneNeighbors( nrt::PointCloud2 const cloud, int const idx, double distance )
{
  int const row = velodyne32::idx2row(idx);
  int const col = velodyne32::idx2col(idx);

  double const rad_per_row = 1.34*M_PI/180.0;
  double const rad_per_col = 0.00287296996;
  int const cloudsize = cloud.size();
  int num_cols = cloudsize / 32;

  float pdist = cloud[idx].getVectorMap().norm();

  // Find the row/col bounds of the search
  int const nv = ceil(atan(distance / pdist) / rad_per_row / 2.0);
  int const nh = ceil(atan(distance / pdist) / rad_per_col / 2.0);
  int const rowmin = std::max(0,row-nv);
  int const rowmax = std::min(31,row+nv);
  int const colmin = col-nh;
  int const colmax = col+nh;

  // Cache a table of the row offsets
  std::array<int, 32> localrowtable;
  for(int irow=rowmin, irow_end=rowmax; irow<=irow_end; ++irow)
    localrowtable[irow-rowmin] = rowTableArray[irow];
  int const * const rowend = &localrowtable[0] + (rowmax-rowmin)+1;

  nrt::Indices indices((colmax-colmin+1)*(rowmax-rowmin+1)+1);
  long unsigned int * indit = &indices[0];
  *indit = idx;
  ++indit;
  for(int icol=colmin, icol_end=colmax; icol<=icol_end; ++icol)
  {
    int *rowit = &localrowtable[0];
    int pcol = (icol < 0) ? (icol+num_cols)*32 : ((icol > num_cols-1) ? (icol - num_cols)*32 : icol*32);
    while(rowit != rowend)
    {
      *indit = pcol + *rowit;
      ++rowit;
      ++indit;
    }
  }

  return indices;
}

void VelodyneNormals::computeFeaturesImpl(
    nrt::PointCloud2 const & cloud,
    nrt::PointCloud2::Iterator<nrt::PointNormal> && iter,
    nrt::PointCloud2::Iterator<nrt::PointNormal> && end )
{
  nrt::PointCloud2::Vector4 planeParams;
  nrt::PointCloud2::Vector3 variation;

  for( ; iter != end; ++iter )
  {
    nrt::Indices neighbors = velodyneNeighbors( cloud, iter.index(), itsDistance );

    auto & normal = iter->get<nrt::PointNormal>();

    if( neighbors.size() < 3 ) // not enough neighbors here
    {
      normal = nrt::PointNormal( std::numeric_limits<nrt::PointCloud2::BaseType>::quiet_NaN() );
      continue;
    }

    // Fit plane to neighbors
    // TODO: This was broken. I made it compile, but I don't think it works... - Rand
    //assert(false);
    nrt::Normals::computePointNormal( cloud, neighbors, itsUseTwoPassCovariance, planeParams, variation );

    // Adjust normal to be oriented according to our viewpoint
    nrt::Normals::faceNormalToViewPoint( iter->geometry(), itsViewPoint, planeParams );

    // Set up final result
    normal = nrt::PointNormal( planeParams[0], planeParams[1], planeParams[2], variation[0] );
  }
}
