/*! @file
    @author Shane Grant
    @copyright GNU Public License (GPL v3)
    @section License
    @verbatim
    // ////////////////////////////////////////////////////////////////////////
    //              The iLab Neuromorphic Robotics Toolkit (NRT)             //
    // Copyright 2010-2012 by the University of Southern California (USC)    //
    //                          and the iLab at USC.                         //
    //                                                                       //
    //                iLab - University of Southern California               //
    //                Hedco Neurociences Building, Room HNB-10               //
    //                    Los Angeles, Ca 90089-2520 - USA                   //
    //                                                                       //
    //      See http://ilab.usc.edu for information about this project.      //
    // ////////////////////////////////////////////////////////////////////////
    // This file is part of The iLab Neuromorphic Robotics Toolkit.          //
    //                                                                       //
    // The iLab Neuromorphic Robotics Toolkit is free software: you can      //
    // redistribute it and/or modify it under the terms of the GNU General   //
    // Public License as published by the Free Software Foundation, either   //
    // version 3 of the License, or (at your option) any later version.      //
    //                                                                       //
    // The iLab Neuromorphic Robotics Toolkit is distributed in the hope     //
    // that it will be useful, but WITHOUT ANY WARRANTY; without even the    //
    // implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       //
    // PURPOSE.  See the GNU General Public License for more details.        //
    //                                                                       //
    // You should have received a copy of the GNU General Public License     //
    // along with The iLab Neuromorphic Robotics Toolkit.  If not, see       //
    // <http://www.gnu.org/licenses/>.                                       //
    // ////////////////////////////////////////////////////////////////////////
    @endverbatim */

#ifdef NRT_HAVE_CLOUD
#ifndef POINTCLOUD_REGISTRATION_REGISTRATIONGICP_H_
#define POINTCLOUD_REGISTRATION_REGISTRATIONGICP_H_

#include <nrt/PointCloud2/PointCloud2.H>
#include <nrt/PointCloud2/Registration/Convergence/ConvergenceCriteriaBase.H>
#include <nrt/PointCloud2/Registration/Correspondence/CorrespondenceEstimationBase.H>
#include <nrt/PointCloud2/Registration/Correspondence/Rejection/CorrespondenceRejectionBase.H>
#include <nrt/PointCloud2/Registration/Transformation/TransformationEstimationBase.H>
#include <PointCloud/Registration/PlaneConstraint.H>

//#include <nrt/PointCloud2/Search/Search.H>

#include <DetectedPlane.H>

struct Covariance
{
  Eigen::Matrix3d value;
  bool valid = false;
};

inline std::ostream & operator<<(std::ostream & os, const Covariance & c)
{
  os << "Covariance: " << std::endl;
  os << "valid? " << c.valid << std::endl;
  os << c.value << std::endl;
  return os;
}


//! Computes the covariances for all points in a cloud
/*! \warning This is currently deprecated. Use computeCovarianceGICP2. */
/*! @return The indices NOT on a plane */
nrt::Indices computeCovarianceGICP( nrt::PointCloud2 & cloud, std::vector<DetectedPlane> & planes ) __attribute__((deprecated));

/*! Computes the covariances for all points in a cloud
    @param[in,out] cloud The point cloud. After calling, the cloud will have
    a valid Covariance structure for each index in the returned set.

    @param[in] zippedPlanes A vector a pairs of (currentPlane, lastPlane) matches.

    @param[in] constraintThreshold The threshold to use when choosing free
    points to add to the returned set. A higher number will result in more
    points returned. This value can be interpreted as the number of points
    required to fulfill each dimension of constraint.
    @return The indices NOT on any plane */
nrt::Indices computeCovarianceGICP2( nrt::PointCloud2 & cloud, std::vector<std::pair<DetectedPlane, DetectedPlane>> zippedPlanes,
    double constraintThreshold = 100, nrt::Optional<PlaneConstraint &> constraint = nrt::OptionalEmpty );

//! A class for registering point clouds using the generalized ICP algorithm
/*! Point cloud registration involves aligning one point cloud to another
    one such that they exist within a common reference frame.

    This class performs registration using the generalized ICP framework, which
    currently cannot fit well into the generic NRT registration framework.

    * Correspondence Method - responsible for choosing which points in a source
                              point cloud correspond to those in a target cloud
    * Correspondence Rejection Criteria - responsible for pruning the correspondences
                                          found between the two clouds.  This can be as
                                          simple as distance, or involve some sort of
                                          sample concensus like RANSAC, or anything else.
    * Rigid Transformation Estimation - fixed to be GICP
    * Convergence Criteria - this evaluates the performance of a rigid transform and
                             determines whether it is good enough to consider convergence
                             as being reached */
class RegistrationGICP
{
  public:
    typedef Eigen::Isometry3f AffineTransform;
    //typedef Eigen::Transform<float, 3, Eigen::Affine> AffineTransform;
    typedef Eigen::Matrix<double, 6, 1> Vector6;

    //! Create a new RegistrationGICP object
    /*! @param[in] correspondenceMethod The method of computing correspondence between points in
                                        two point clouds.
        @param[in] correspondenceRejectionCriteria An initial criteria for rejecting correspondences found
                                                   via correspondenceMethod.
        @param[in] convergeCriteria The method of determining convergence for the transformation between
                                    the two clouds */
    RegistrationGICP( nrt::CorrespondenceEstimationBase::SharedPtr const correspondenceMethod,
                      std::vector<nrt::CorrespondenceRejectionBase::SharedPtr> const & correspondenceRejectionCriteria,
                      nrt::ConvergenceCriteriaBase::SharedPtr const convergeCriteria );

    //! Aligns a source cloud to a target cloud
    /*! The details of how this alignment is performed are specified by the parameters
        given during construction of the RegistrationGICP object.

        @param source The cloud that will be aligned
        @param sourceIndices The indices that are not on planes
        @param target The target cloud the source will be aligned to
        @param guess An initial guess for the alignment
        @return The transformation that aligns the source with the target */
    AffineTransform align( nrt::PointCloud2 const & source, nrt::Indices const & sourceIndices,
                           nrt::PointCloud2 const & target, nrt::Indices const & targetIndices,
                           std::vector<std::pair<DetectedPlane, DetectedPlane>> sourceTargetPlanes,
                           AffineTransform const & guess = AffineTransform::Identity() );

    //! Returns true if the most recent alignment converged
    bool converged() const;

  protected:
    void bfgs( nrt::PointCloud2::ConstIterator<> && sourceIter,
        nrt::PointCloud2::ConstIterator<> && targetIter,
        std::vector<Eigen::Matrix3d> const & mahalanobis,
        size_t numInputs,
        AffineTransform const & guess,
        AffineTransform & transform );

  private:
    nrt::CorrespondenceEstimationBase::SharedPtr itsCorrespondenceEstimation;
    std::vector<nrt::CorrespondenceRejectionBase::SharedPtr> itsCorrespondenceRejection;
    nrt::ConvergenceCriteriaBase::SharedPtr itsConvergenceCriteria;

    bool itsConverged;
};

#endif // POINTCLOUD_REGISTRATION_REGISTRATIONGICP_H_
#endif // NRT_HAVE_CLOUD
