#include "OctomapUtils.H"
#include <nrt/ImageProc/Drawing/ColorMapping.H>

using namespace nrt;

#ifdef USE_OCTOMAP
#include <octomap/OcTree.h>

void drawOctree(std::shared_ptr<graphics::ShapeRendererBasic> renderer,
    std::vector<Eigen::Isometry3d> const & transformHistory, std::vector<PointCloud2> const & cloudHistory)
   
{
  static graphics::PointCloud cloudGraphics;
  static size_t lastCloudSize = 0;

  float const scale = 100.0f;

  if(lastCloudSize != cloudHistory.size())
  {
    static octomap::OcTree octree(.1);
    octree.setOccupancyThres(0.75);
    octree.setClampingThresMax(1.0);
    octree.setClampingThresMin(0.0);

    lastCloudSize = cloudHistory.size();

    NRT_INFO("Calculating Octree");
    for(size_t i=0; i<cloudHistory.size(); ++i)
    {
      NRT_INFO(" >>> " << i+1 << "/" << cloudHistory.size());

      nrt::PointCloud2 const & cloud = cloudHistory[i];
      Eigen::Affine3f tform(transformHistory[i].cast<float>().inverse());

      octomap::point3d origin(
          transformHistory[i].translation().x()*scale, 
          transformHistory[i].translation().y()*scale, 
          transformHistory[i].translation().z()*scale
          );

      for(auto p : cloud)
      {
        Eigen::Vector3f const pt = (tform * p.geometry().getVector3Map()) * scale;
        octomap::point3d point(pt.x(), pt.y(), pt.z());
        //octree.insertRay(origin, point, -1.0, true);
        octree.updateNode(pt.x(), pt.y(), pt.z(), 1.0f, true);
      }
    }
    octree.updateInnerOccupancy();

    NRT_INFO("Octree has " << octree.size() << " elements");

    if(octree.size() == 0) return;

    PointCloud2 octreeCloud;
    int const maxDepth = 8;

    float maxHeight = std::numeric_limits<float>::lowest();
    float minHeight = std::numeric_limits<float>::max();
    for(auto it = octree.begin_leafs(maxDepth), end = octree.end_leafs(); it != end; ++it)
    {
      if(octree.isNodeOccupied(*it))
      {
        float s = 1.0 / scale;
        float const z = float(it.getZ())*s;
        octreeCloud.insert({ float(it.getX())*s, float(it.getY())*s, z });
        maxHeight = std::max(z, maxHeight);
        minHeight = std::min(z, minHeight);
      }
    }

    octreeCloud.addField<PixRGB<uint8_t>>();
    for(auto p : octreeCloud.range<PixRGB<uint8_t>>())
    {
      p.get<PixRGB<uint8_t>>() = PixRGB<uint8_t>(jetPixel<float>(p.geometry().z(), minHeight, maxHeight));
    }

    NRT_INFO("Octree Cloud has " << octreeCloud.size() << " points of size " << octree.begin_leafs(maxDepth).getSize());

    cloudGraphics = graphics::PointCloud(octreeCloud);
    cloudGraphics.setDisplayType(graphics::PointCloud::Cube);
    cloudGraphics.setPointSize(octree.begin_leafs(maxDepth).getSize() / scale * .4);
  }

  cloudGraphics.render(*renderer);
}

#else

void drawOctree(std::shared_ptr<graphics::ShapeRendererBasic> renderer,
    std::vector<Eigen::Isometry3d> const & transformHistory, std::vector<PointCloud2> const & cloudHistory)
   
{
  std::cerr << "--- You did not compile against liboctomap!" << std::endl;
}

#endif
