#include "PoseToPlaneFactor.H"
#include <gtsam/geometry/Pose3.h>
#include <gtsam/geometry/Point3.h>
#include <gtsam/nonlinear/Symbol.h>
#include <gtsam/slam/PriorFactor.h>
#include <gtsam/slam/BetweenFactor.h>
#include <gtsam/nonlinear/NonlinearFactorGraph.h>
#include <gtsam/nonlinear/ISAM2.h>
#include <gtsam/nonlinear/Values.h>
#include <gtsam/nonlinear/NonlinearFactor.h>

#include <random>

#define UPDATE_ISAM(frame) \
  isam.update(graph, initialEstimate); \
  currentEstimate = isam.calculateBestEstimate(); \
  std::cout << "======================================" << std::endl; \
  currentEstimate.print("Frame " + std::to_string(frame) + " Estimate:\n"); \
  graph.resize(0); \
  initialEstimate.clear(); \

const static bool GROUND_TRUTH = false;
const static double SIGMA_A = 0.1;
const static double SIGMA_T = 0.05;

double ga()
{
  static std::random_device rd;
  static std::mt19937 gen(rd());
  static std::normal_distribution<> d(0,(GROUND_TRUTH) ? 0. : SIGMA_A);

  return d(gen);
}

double gt()
{
  static std::random_device rd;
  static std::mt19937 gen(rd());
  static std::normal_distribution<> d(0,(GROUND_TRUTH) ? 0. : SIGMA_T);

  return d(gen);
}

int main()
{
  using namespace gtsam;

  std::vector<Eigen::Hyperplane<double, 3>> planes = {
    {{-1, 0, 0}, 10}, {{0, -1, 0}, 10}, {{0, 0, -1}, 10}
  };

  std::vector<Eigen::Isometry3d> poses = {Eigen::Isometry3d::Identity()};
  std::vector<Eigen::Isometry3d> odometry;

  { // Forwards
    int steps = 10;
    int distance = 7;
    for(int i=0; i<steps; ++i)
    {
      odometry.emplace_back(Eigen::Translation3d(0, double(distance) / double(steps), 0));
      poses.emplace_back(poses.back() * odometry.back());
    }
  }

  { // 90 Degree Left Turn
    int steps = 50;
    for(int i=0; i<steps; ++i)
    {
      odometry.emplace_back(Eigen::AngleAxisd(M_PI/2.0/double(steps), Eigen::Vector3d::UnitZ()));
      poses.emplace_back(poses.back() * odometry.back());
    }
  }

  { // Forwards
    int steps = 10;
    int distance = 7;
    for(int i=0; i<steps; ++i)
    {
      odometry.emplace_back(Eigen::Translation3d(0, double(distance) / double(steps), 0));
      poses.emplace_back(poses.back() * odometry.back());
    }
  }

  ISAM2Params parameters;
  parameters.relinearizeThreshold = 0.01;
  parameters.relinearizeSkip = 1;
  ISAM2 isam(parameters);

  NonlinearFactorGraph graph;
  Values initialEstimate;
  Values currentEstimate;

  noiseModel::Diagonal::shared_ptr priorNoise       = noiseModel::Diagonal::Sigmas((Vector(6) << SIGMA_A, SIGMA_A, SIGMA_A, SIGMA_T, SIGMA_T, SIGMA_T));
  noiseModel::Diagonal::shared_ptr odometryNoise    = noiseModel::Diagonal::Sigmas((Vector(6) << SIGMA_A, SIGMA_A, SIGMA_A, SIGMA_T, SIGMA_T, SIGMA_T));
  noiseModel::Diagonal::shared_ptr measurementNoise = noiseModel::Diagonal::Sigmas((Vector(4) << SIGMA_T, SIGMA_T, SIGMA_T, SIGMA_T));


  // Insert the initial plane estimates
  for(size_t i=0; i<planes.size(); ++i)
    initialEstimate.insert(gtsam::Symbol('z', i), PlaneValue(planes[i]));

  int idx=0;
  for(Eigen::Isometry3d const & pose : poses)
  {
    for(size_t planeIdx=0; planeIdx<planes.size(); ++planeIdx)
    {
      Eigen::Hyperplane<double, 3> plane = planes[planeIdx];
      Eigen::Hyperplane<double, 3> z = plane.transform(Eigen::Affine3d(pose.inverse()));
      graph.add(PoseToPlaneFactor(PlaneValue(z), measurementNoise, gtsam::Symbol('p', idx), gtsam::Symbol('z', planeIdx)));
    }

    initialEstimate.insert(gtsam::Symbol('p', idx), Pose3(pose.matrix()));

    if(idx == 0)
      graph.add(PriorFactor<Pose3>(gtsam::Symbol('p', 0), Pose3(Rot3::identity(), Point3(0+gt(),0+gt(),0+gt())), priorNoise));
    else
      graph.add(BetweenFactor<Pose3>(gtsam::Symbol('p', idx-1), gtsam::Symbol('p', idx), Pose3(odometry[idx].matrix()), odometryNoise));

    UPDATE_ISAM(idx);

    ++idx;
  }

























  //Symbol p0('p', 0), p1('p', 1), p2('p', 2), p3('p', 3);
  //Symbol m0('m', 0), m1('m', 1), m2('m', 2), m3('m', 3);

  //NonlinearFactorGraph graph;
  //Values initialEstimate;
  //Values currentEstimate;

  //ISAM2Params parameters;
  //parameters.relinearizeThreshold = 0.01;
  //parameters.relinearizeSkip = 1;
  //ISAM2 isam(parameters);

  //noiseModel::Diagonal::shared_ptr priorNoise       = noiseModel::Diagonal::Sigmas(Vector_(6, SIGMA_A, SIGMA_A, SIGMA_A, SIGMA_T, SIGMA_T, SIGMA_T));
  //noiseModel::Diagonal::shared_ptr odometryNoise    = noiseModel::Diagonal::Sigmas(Vector_(6, SIGMA_A, SIGMA_A, SIGMA_A, SIGMA_T, SIGMA_T, SIGMA_T));
  //noiseModel::Diagonal::shared_ptr measurementNoise = noiseModel::Diagonal::Sigmas(Vector_(4, SIGMA_T, SIGMA_T, SIGMA_T, SIGMA_T));

  //graph.add(PriorFactor<Pose3>(p0, Pose3(Rot3::identity(), Point3(0+gt(),0+gt(),0+gt())), priorNoise));
  //graph.add(PoseToPlaneFactor(PlaneValue(-1+gt(), 0+gt(), 0+gt(), 10+gt()), measurementNoise, p0, m0));
  //graph.add(PoseToPlaneFactor(PlaneValue(0+gt(), 0+gt(), -1+gt(), 10+gt()), measurementNoise, p0, m2));
  //
  //initialEstimate.insert(p0, Pose3(Rot3::Rz(ga()), Point3(gt(), gt(), gt()))); // gt
  //initialEstimate.insert(m0, PlaneValue(-1 + gt(), 0 + gt(), 0 + gt(), 10 + gt())); // gt
  //initialEstimate.insert(m2, PlaneValue(0 + gt(), 0 + gt(), -1 + gt(), 10 + gt())); // gt

  //UPDATE_ISAM(0)

  //graph.add(BetweenFactor<Pose3>(p0, p1, Pose3(Rot3::Rz(M_PI/4.0+ga()), Point3(2+gt(), 2+gt(), 0+gt())), odometryNoise));
  //graph.add(PoseToPlaneFactor(PlaneValue(-1./std::sqrt(2.)+gt(), 1./std::sqrt(2.)+gt(), 0+gt(), 8+gt()), measurementNoise, p1, m0));
  //graph.add(PoseToPlaneFactor(PlaneValue(0+gt(), -1+gt(), 0+gt(), 10+gt()), measurementNoise, p1, m1));
  //graph.add(PoseToPlaneFactor(PlaneValue(0+gt(), 0+gt(), -1+gt(), 10+gt()), measurementNoise, p1, m2));

  //initialEstimate.insert(p1, Pose3(Rot3::Rz(M_PI/4.0 + ga()), Point3(2 + gt(), 2 + gt(), 0 + gt()))); // gt
  //initialEstimate.insert(m1, PlaneValue(1./std::sqrt(2.) + gt(), -1./std::sqrt(2.) + gt(), 0 + gt(), 10 + gt())); // gt

  //UPDATE_ISAM(1)

  //graph.add(BetweenFactor<Pose3>(p1, p2, Pose3(Rot3::Rz(-M_PI/4.0+ga()), Point3(0+gt(), 2.*std::sqrt(2.)+gt(), 0+gt())), odometryNoise));
  //graph.add(PoseToPlaneFactor(PlaneValue(-1+gt(), 0+gt(), 0+gt(), 10+gt()), measurementNoise, p2, m0));
  //graph.add(PoseToPlaneFactor(PlaneValue(0+gt(), 0+gt(), -1+gt(), 10+gt()), measurementNoise, p2, m2));
  //
  //// ground truth
  //initialEstimate.insert(p2, Pose3(Rot3::Rz(ga()), Point3(0 + gt(), 4 + gt(), 0 + gt()))); // gt

  //UPDATE_ISAM(2)

  //graph.add(BetweenFactor<Pose3>(p2, p3, Pose3(Rot3::Rz(M_PI/2.+ga()), Point3(-2+gt(), 0+gt(), 0+gt())), odometryNoise));
  //graph.add(PoseToPlaneFactor(PlaneValue(-1./std::sqrt(2.)+gt(), -1./std::sqrt(2.)+gt(), 0+gt(), 5.7573593128588+gt()), measurementNoise, p3, m1));
  //graph.add(PoseToPlaneFactor(PlaneValue(0+gt(), 0+gt(), -1+gt(), 10+gt()), measurementNoise, p3, m2));

  //// ground truth
  //initialEstimate.insert(p3, Pose3(Rot3::Rz(M_PI/2. + ga()), Point3(-2 + gt(), 4 + gt(), 0 + gt()))); // gt
  //UPDATE_ISAM(3)


  return 0;
}

