#pragma once

#include <gtsam/nonlinear/NonlinearFactor.h>
#include <gtsam/geometry/Pose3.h>
#include <gtsam/nonlinear/Symbol.h>
#include <gtsam/geometry/Point3.h>

class PlaneValue : public gtsam::DerivedValue<PlaneValue>
{
  public:
    typedef Eigen::Vector4d Vector4;
    typedef Eigen::Vector3d Vector3;
    typedef typename Vector3::Scalar Scalar;

    static const size_t dimension = 4;

    PlaneValue() : data(Vector4::Zero()) { n()[0] = 1; }

    PlaneValue(double nx, double ny, double nz, double d) :
      data(nx, ny, nz, d) { n().normalize(); }

    PlaneValue(Vector4 const & v) : data(v) { n().normalize(); }

    PlaneValue(Eigen::Hyperplane<double, 3> const & p)
    {
      n() = p.normal();
      d() = p.offset();
    }

    void print(std::string const & s = "") const override
    { std::cout << s << "[" << data.transpose() << "]" << std::endl; }

    size_t dim() const override
    { return dimension; }

    PlaneValue retract(gtsam::Vector const & v) const
    {
      PlaneValue ret(data + v);
      ret.n().normalize();
      return ret;
    }

    gtsam::Vector localCoordinates(PlaneValue const & p) const
    {
      gtsam::Vector ret = p.data - this->data;
      ret.block<3,1>(0,0).normalize();
      return ret;
    }

    bool operator==(PlaneValue const & p) const
    { return this->data == p.data; }

    bool equals(PlaneValue const & p, double tol) const
    { return this->data.isApprox(p.data, tol); }

    Eigen::Block<Vector4,3,1> n()
    { return data.block<3,1>(0,0); }

    Eigen::Block<Vector4 const,3,1> n() const
    { return data.block<3,1>(0,0); }

    double & d()
    { return data[3]; }

    double const & d() const
    { return data[3]; }

  private:
    Vector4 data;
};

#if GTSAM_VERSION_MAJOR == 4
struct PlaneValueTraits : public gtsam::Testable<PlaneValue>
{
  static int GetDimension( PlaneValue const & )
  {
    return PlaneValue::dimension;
  }

  static PlaneValue Retract( PlaneValue const & origin, gtsam::Vector const & v )
  {
    return origin.retract( v );
  }

  static gtsam::Vector Local( PlaneValue const & origin, PlaneValue const & other )
  {
    return origin.localCoordinates( other );
  }
};

namespace gtsam
{
  template<> struct traits<PlaneValue> : public PlaneValueTraits {};
}
#endif
