#ifndef ALGORITHMS_POINTCLOUD_FEATURES_FEATURECOMPONENTBASE_H_
#define ALGORITHMS_POINTCLOUD_FEATURES_FEATURECOMPONENTBASE_H_

#include <nrt/Core/Model/Component.H>
#include <nrt/Core/Design/Factory.H>
#include <nrt/PointCloud2/PointCloud2.H>

//! Base component for all features
/*! This class can be thought of as a component wrapper around
    nrt::FeatureBase in PointCloud2, and follows the same internal
    workings.
    
    Derived classes should implement the static functions id() and
    description(), both returning an std::string.  id() gives a name to the
    feature, and description() provides a description.
    */
class FeatureComponentBase : public nrt::Component
{
  public:
    FeatureComponentBase( std::string const & instanceID = "FeatureComponentBase" );
    
    //! Extract features from the input point cloud
    /*! @param input The input point cloud
        @return The input with the new feature */
    virtual nrt::PointCloud2 computeFeature( nrt::PointCloud2 const input ) = 0;

    //! Extract features on a subset of an input point cloud
    /*! @param input The input point cloud
        @param indices The indices describing a subset of the cloud
        @return The input with the new feature */
    virtual nrt::PointCloud2 computeFeature( nrt::PointCloud2 const input, nrt::Indices const indices ) = 0;
};

typedef nrt::Factory<
FeatureComponentBase,
std::string,
std::function<std::shared_ptr<FeatureComponentBase>()>,
std::map<std::string, std::string>
> PointCloudFeatureFactory;

/*! \def NRTBASE_REGISTER_POINTCLOUDFEATURE(CLASSNAME)
  Register a PointCloudFeatureType with a global Factory.
*/
#define NRTBASE_REGISTER_POINTCLOUDFEATURE(CLASSNAME)                         \
    std::shared_ptr<FeatureComponentBase> create##CLASSNAME()            			\
    { return std::shared_ptr<FeatureComponentBase>(new CLASSNAME); }     			\
    bool pointCloudFeatureRegister##CLASSNAME()                               \
    {                                                                         \
      std::map<std::string,std::string> metadata;                             \
      metadata["id"] = CLASSNAME::id();                                       \
      metadata["description"] = CLASSNAME::description();                     \
      PointCloudFeatureFactory::instance().registerType(                     	\
          metadata["id"],                                                     \
          create##CLASSNAME,                                                  \
          metadata                                                            \
      );                                                                      \
      return true;                                                            \
    }                                                                         \
    const bool CLASSNAME##Registered __attribute__ ((unused)) =               \
                                        pointCloudFeatureRegister##CLASSNAME();
 
#endif // ALGORITHMS_POINTCLOUD_FEATURES_FEATURECOMPONENTBASE_H_
