#include "FindTransforms.H"
#include "details/LinearAlgebra.H"
#include <Config/Config.H>

NRT_BEGIN_UNCHECKED_INCLUDES;
#include <Eigen/Eigenvalues>
NRT_END_UNCHECKED_INCLUDES;

namespace
{
  // ######################################################################
  //! Transforms from quaternion covariance to Euler angle covariance
  /*! Taken from pages 69-70 of "Development of a Real-Time Attitude System
      Using a Quaternion Parameterization and Non-Dedicated GPS Receivers"
      by John B. Schleppe, 1996

      @param q The quaternion
      @param Cq the quaternion covariance
      */
  Eigen::Matrix3d quaternionToEulerCovariance( Eigen::Quaterniond const & q, Eigen::Matrix4d const & Cq )
  {
    const double q1 = q.x();
    const double q2 = q.y();
    const double q3 = q.z();
    const double q4 = q.w();

    // Compute G matrix coefficients
    const double denomLeft  = (q3 + q2) * (q3 + q2) + (q4 + q1) * (q4 + q1);
    const double denomRight = (q3 - q2) * (q3 - q2) + (q4 - q1) * (q4 - q1);

    // Phi portion
    const double dPhi_dq1 = -(q3 + q2) / denomLeft + (q3 - q2) / denomRight;
    const double dPhi_dq2 =  (q4 + q1) / denomLeft - (q4 - q1) / denomRight;
    const double dPhi_dq3 =  (q4 + q1) / denomLeft + (q4 - q1) / denomRight;
    const double dPhi_dq4 = -(q3 + q2) / denomLeft - (q3 - q2) / denomRight;

    // Theta portion
    const double thetaDenom = std::sqrt( 1 - 4 * (q2*q3 + q1*q4) * (q2*q3 + q1*q4) );

    const double dThe_dq1 = 2 * q4 / thetaDenom;
    const double dThe_dq2 = 2 * q3 / thetaDenom;
    const double dThe_dq3 = 2 * q2 / thetaDenom;
    const double dThe_dq4 = 2 * q1 / thetaDenom;

    // Psi portion
    const double dPsi_dq1 = -(q3 + q2) / denomLeft - (q3 - q2) / denomRight;
    const double dPsi_dq2 =  (q4 + q1) / denomLeft + (q4 - q1) / denomRight;
    const double dPsi_dq3 =  (q4 + q1) / denomLeft - (q4 - q1) / denomRight;
    const double dPsi_dq4 = -(q3 + q2) / denomLeft + (q3 - q2) / denomRight;

    // Constuct G
    Eigen::Matrix<double, 3, 4> G;
    G << dPhi_dq1, dPhi_dq2, dPhi_dq3, dPhi_dq4,
         dThe_dq1, dThe_dq2, dThe_dq3, dThe_dq4,
         dPsi_dq1, dPsi_dq2, dPsi_dq3, dPsi_dq4;

    return G * Cq * G.transpose();
  }
} // anon namespace

// ######################################################################
Eigen::Translation3f findTranslation(nrt::Correspondences const & correspondences, std::vector<DetectedPlane> const & frame1Planes,
    std::vector<DetectedPlane> const & frame2Planes,
    nrt::Optional<int&> rnk)
{
  if(correspondences.size() < 3)
  {
    if(rnk) *rnk = 0;
    return Eigen::Translation3f::Identity();
  }

  Eigen::MatrixXd C_d = Eigen::MatrixXd::Zero(correspondences.size(), correspondences.size());
  Eigen::MatrixXd M(correspondences.size(), 3);
  Eigen::VectorXd d(correspondences.size());
  for(size_t i=0; i<correspondences.size(); ++i)
  {
    DetectedPlane const & a = frame1Planes[correspondences[i].sourceIndex];
    DetectedPlane const & b = frame2Planes[correspondences[i].targetIndex];

    M.block<1,3>(i,0) = a.plane.normal().transpose().cast<double>();
    d[i] = b.plane.offset() - a.plane.offset();

    C_d(i, i) = a.rhoCovariance * b.rhoCovariance;

    Eigen::Matrix4d const & H_a = a.hessian;
    Eigen::Matrix4d const C_a = -pseudoInverse(H_a, 10e9*std::numeric_limits<double>::epsilon());

    Eigen::Matrix4d const & H_b = b.hessian;
    Eigen::Matrix4d const C_b = -pseudoInverse(H_b, 10e9*std::numeric_limits<double>::epsilon());

    C_d(i, i) = std::abs(C_a.trace()) + std::abs(C_b.trace());
  }

  Eigen::DiagonalMatrix<double, Eigen::Dynamic> W(C_d.diagonal().cwiseInverse().cwiseSqrt());

  Eigen::MatrixXd const M_hat = W*M;
  Eigen::VectorXd const d_hat = W*d;

  Eigen::JacobiSVD<Eigen::MatrixXd> svd(M_hat, Eigen::ComputeFullU | Eigen::ComputeFullV);
  Eigen::MatrixXd U = svd.matrixU();
  Eigen::Vector3d S = svd.singularValues();
  Eigen::MatrixXd V = svd.matrixV();

  double const c = 400;
  double const tol = 10e-7;
  size_t rank = 0;
  if( S[0] > tol)
  {
    rank = 1;
    rank = rank + (S[1] > S[0]/c);
    rank = rank + (S[2] > S[1]/c);
  }

  Eigen::Vector3d t(0,0,0);
  for(size_t i=0; i<rank; ++i)
  {
    Eigen::VectorXd u_i_hat = U.block(0, i, U.rows(), 1);
    u_i_hat = u_i_hat / u_i_hat.norm();

    Eigen::VectorXd v_i_hat = V.block<3, 1>(0, i);
    v_i_hat = v_i_hat / v_i_hat.norm();

    t += (1.0 / S[i]) * u_i_hat.dot(d_hat) * v_i_hat;
  }

  if(rnk) *rnk = rank;

  return Eigen::Translation3f(t.cast<float>());
}

// ######################################################################
Eigen::Matrix3f findRotation(nrt::Correspondences const & correspondences,
    std::vector<DetectedPlane> const & frame1Planes, std::vector<DetectedPlane> const & frame2Planes,
    nrt::Optional<Eigen::Matrix4d&> covariance, nrt::Optional<double&> rotationUncertainty)
{
  Eigen::Matrix4d K;

  for(nrt::Correspondence const & c : correspondences)
  {
    DetectedPlane const & f1 = frame1Planes[c.sourceIndex];
    DetectedPlane const & f2 = frame2Planes[c.targetIndex];

    Eigen::Vector3d n1 = f1.plane.normal().cast<double>();
    Eigen::Vector3d n2 = f2.plane.normal().cast<double>();

    auto const & hnn1 = f1.hessian.block<3,3>(0,0);
    auto const & hnd1 = f1.hessian.block<3,1>(1,0);
    double const hdd1 = f1.hessian(3,3);

    auto const & hnn2 = f2.hessian.block<3,3>(0,0);
    auto const & hnd2 = f2.hessian.block<3,1>(1,0);
    double const hdd2 = f2.hessian(3,3);

    Eigen::Matrix3d xxx = hnn1 - hnd1 * (1.0 / hdd1) * hnd1.transpose();
    Eigen::Matrix3d yyy = hnn2 - hnd2 * (1.0 / hdd2) * hnd2.transpose();
    auto ncov1 = -pseudoInverse(xxx, 10e7*std::numeric_limits<double>::epsilon());
    auto ncov2 = -pseudoInverse(yyy, 10e7*std::numeric_limits<double>::epsilon());

    double const w = 1.0 / (ncov1.trace() + ncov2.trace());

    auto crossmat = [](Eigen::Vector3d const & x)
    {
      Eigen::Matrix3d m;
      m <<  0,    -x(2),  x(1),
            x(2),  0,    -x(0),
           -x(1),  x(0),  0;
      return m;
    };

    Eigen::Matrix4d psi;
    psi(0,0) = 0;
    psi.block<1,3>(0,1) = -n1.transpose();
    psi.block<3,1>(1,0) =  n1;
    psi.block<3,3>(1,1) =  crossmat(n1);

    Eigen::Matrix4d psi_bar;
    psi_bar(0,0) = 0;
    psi_bar.block<1,3>(0,1) = -n2.transpose();
    psi_bar.block<3,1>(1,0) =  n2;
    psi_bar.block<3,3>(1,1) =  -crossmat(n2);

    K += w * psi_bar.transpose() * psi;
  }

  Eigen::SelfAdjointEigenSolver<Eigen::Matrix4d> eigensolver(K);
  auto eigenVector = eigensolver.eigenvectors().block<4,1>(0,3);
  double eigenValue = eigensolver.eigenvalues()(0);

  Eigen::Quaterniond result;
  result.w() = eigenVector[0];
  result.x() = eigenVector[1];
  result.y() = eigenVector[2];
  result.z() = eigenVector[3];

  if(covariance || rotationUncertainty)
  {
    Eigen::Matrix4d Hqq = 2*(K-eigenValue*Eigen::Matrix4d::Identity());
    Eigen::Matrix4d cov = -pseudoInverse(Hqq);
    if(covariance) *covariance = cov;

    if(rotationUncertainty)
    {
      Eigen::SelfAdjointEigenSolver<Eigen::Matrix4d> eigensolver(cov);
      Eigen::Vector4d eigenValues = eigensolver.eigenvalues();
      double product = 1.0;
      for(size_t i=0; i<4; ++i)
        if(eigenValues[i] != 0)
          product *= eigenValues[i];
      *rotationUncertainty = std::abs(product);
    }
  }

  return result.matrix().cast<float>();
}

// ######################################################################
Eigen::Isometry3d findTransform(nrt::Correspondences const & correspondences,
    std::vector<DetectedPlane> const & frame1Planes, std::vector<DetectedPlane> const & frame2Planes,
    nrt::Optional<Eigen::Matrix3d&> translationCovariance,
    nrt::Optional<Eigen::Matrix3d&> rotationCovariance
    )
{
  static double maxTranslationCov = config::lookup<double>("correspondences.transformestimate.maxTranslationCov");

  //Eigen::Vector3f translation = findTranslation(correspondences, frame1Planes, frame2Planes, nrt::OptionalEmpty).inverse().vector();

  Eigen::Vector3f translation = pathak::findTranslation(correspondences, frame1Planes, frame2Planes,
      nrt::OptionalEmpty, nrt::OptionalEmpty, translationCovariance).vector();

  for(size_t i=0; i<3; ++i)
    if((*translationCovariance)(i,i) > maxTranslationCov || std::abs(translation[i]) > 0.5)
    {
      translation[i] = 0.0;
    }

  Eigen::Matrix3f rotation = pathak::findRotation(correspondences, frame1Planes, frame2Planes, rotationCovariance);

  auto transform = Eigen::Isometry3d::Identity();
  transform.rotate(rotation.cast<double>());
  transform.translation() = translation.cast<double>();

  return transform;
}

namespace pathak
{
  // ######################################################################
  //! Find the effective rank of the SVD decomposition.
  /*! As a side-effect, set all elements of S to zero that are rank deficient */
  int effectiveRank(Eigen::MatrixXd const & U, Eigen::Vector3d & S, Eigen::MatrixXd const & V,
      double const maxConditionNumber = 200.0)
  {
    double const minMaxSV = 1e-7;

    int rank = 3;

    if(S[0] > minMaxSV)
    {
      double epsilonZero;
      if(maxConditionNumber <= std::numeric_limits<double>::epsilon())
        epsilonZero = std::numeric_limits<double>::epsilon();
      else
        epsilonZero = S[0] / maxConditionNumber;

      for(int i=0; i<3; ++i)
      {
        if(S[i] <= epsilonZero)
        {
          --rank;
          S[i] = 0;
        }
      }
    }
    else
    {
      rank = 0;
    }

    return rank;
  }
}


// ######################################################################
Eigen::Matrix3f pathak::findRotation(nrt::Correspondences const & correspondences,
    std::vector<DetectedPlane> const & frame1Planes, std::vector<DetectedPlane> const & frame2Planes,
    nrt::Optional<Eigen::Matrix3d&> covariance, nrt::Optional<double&> uncertaintyVolume)
{
  std::vector<double> ws;
  double wSum = 0;
  Eigen::Matrix3d S = Eigen::Matrix3d::Zero();

  for(nrt::Correspondence const & c : correspondences)
  {
    DetectedPlane const & p1 = frame1Planes[c.sourceIndex];
    DetectedPlane const & p2 = frame2Planes[c.targetIndex];

    double const w = 1.0 / (p1.normalCovariance.trace() + p2.normalCovariance.trace());
    ws.push_back(w);
    wSum += w;

    S += w * p1.plane.normal().cast<double>() * p2.plane.normal().transpose().cast<double>();
  }

  Eigen::Matrix4d N;
	N(0,0)= S(0,0)+S(1,1)+S(2,2);
	N(0,1)= S(1,2)-S(2,1);
	N(0,2)= S(2,0)-S(0,2);
	N(0,3)= S(0,1)-S(1,0);
	N(1,0)= N(0,1);
	N(1,1)= S(0,0)-S(1,1)-S(2,2);
	N(1,2)= S(0,1)+S(1,0);
	N(1,3)= S(2,0)+S(0,2);
	N(2,0)= N(0,2);
	N(2,1)= N(1,2);
	N(2,2)= -S(0,0)+S(1,1)-S(2,2);
	N(2,3)= S(1,2)+S(2,1);
	N(3,0)= N(0,3);
	N(3,1)= N(1,3);
	N(3,2)= N(2,3);
	N(3,3)= -S(0,0)-S(1,1)+S(2,2);

  Eigen::SelfAdjointEigenSolver<Eigen::Matrix4d> eigensolver(N);
  Eigen::Vector4d const maxEigenVector = eigensolver.eigenvectors().block<4,1>(0,3);
  double const maxEigenValue = eigensolver.eigenvalues()[3];

  Eigen::Quaterniond q(maxEigenVector[0], maxEigenVector[1], maxEigenVector[2], maxEigenVector[3]);
  q.normalize();

  double const factor = 2.0;
  Eigen::Matrix4d H = (N - Eigen::Matrix4d::Identity() * maxEigenValue) * factor;
  Eigen::Matrix4d infoMatrix = -H;

  if(covariance || uncertaintyVolume)
  {
    // Find the generalized inverse of the info matrix
    eigensolver.compute(infoMatrix);
    Eigen::Matrix4d const eigenVectors = eigensolver.eigenvectors();
    Eigen::Vector4d const eigenValues  = eigensolver.eigenvalues();

    double infoDeterminant = 1.0;
    Eigen::Matrix4d Info_inv = Eigen::Matrix4d::Zero();
    for(int i=1; i<eigenValues.size(); ++i)
    {
      Info_inv += eigenVectors.col(i) * eigenVectors.col(i).transpose() / eigenValues[i];
      infoDeterminant *= 1.0 / eigenValues[i];
    }

    if(covariance)
    {
      *covariance = quaternionToEulerCovariance( q, Info_inv );
      //std::cout << *covariance << std::endl;
    }

    if(uncertaintyVolume) *uncertaintyVolume = std::abs(infoDeterminant);
  }

  return q.matrix().cast<float>();
}

// ######################################################################
Eigen::Vector3d pathak::findTranslationByOverlap(nrt::Correspondences const & correspondences,
    std::vector<DetectedPlane> const & frame1Planes, std::vector<DetectedPlane> const & frame2Planes,
    Eigen::Matrix3d const & rotationEstimate, nrt::Optional<Eigen::Matrix3d&> covariance)
{
  Eigen::Matrix3d C_sum     = Eigen::Matrix3d::Zero();
  Eigen::Matrix3d A         = Eigen::Matrix3d::Zero();
  Eigen::Vector3d b         = Eigen::Vector3d::Zero();

  for(size_t i=0; i<correspondences.size(); ++i)
  {
    DetectedPlane const & p1 = frame1Planes[correspondences[i].sourceIndex];
    DetectedPlane const & p2 = frame2Planes[correspondences[i].targetIndex];
    assert(p1.centroid && p2.centroid);
    assert(p1.pointCovariance && p2.pointCovariance);

    Eigen::Vector3d const & c1 = *p1.centroid;
    Eigen::Vector3d const & c2 = *p2.centroid;

    Eigen::Matrix3d const & cov1 = *p1.pointCovariance;
    Eigen::Matrix3d const & cov2 = *p2.pointCovariance;

    Eigen::Vector3d t_i = c1 - rotationEstimate * c2;

    Eigen::Matrix3d Ci_t = cov1 + rotationEstimate * cov2 * rotationEstimate.transpose();
    C_sum += Ci_t;

    Eigen::Matrix3d Ci_t_inv = Ci_t.inverse();

    A += Ci_t_inv;
    b += Ci_t_inv * t_i;
  }

  Eigen::Vector3d const translation = A.inverse() * b;
  if(covariance) *covariance = C_sum;

  return translation;
}

// ######################################################################
Eigen::Translation3f pathak::findTranslation(nrt::Correspondences const & correspondences,
    std::vector<DetectedPlane> const & frame1Planes, std::vector<DetectedPlane> const & frame2Planes,
    nrt::Optional<Eigen::Matrix3d const &> rotationEstimate,
    nrt::Optional<int&> rank_, nrt::Optional<Eigen::Matrix3d&> covariance_, nrt::Optional<double&> translationUncertainty_)
{
  // Output params
  if(rank_) *rank_ = 0;
  if(covariance_) *covariance_ = Eigen::Matrix3d::Identity() * 1.0; // std::numeric_limits<double>::infinity();
  if(translationUncertainty_) *translationUncertainty_ = 1; //std::numeric_limits<double>::infinity();

  if( correspondences.size() < 3 )
    return Eigen::Translation3f::Identity();

  Eigen::MatrixXd M_hat(correspondences.size(), 3);
  Eigen::VectorXd d_hat(correspondences.size());
  for(size_t i=0; i<correspondences.size(); ++i)
  {
    DetectedPlane const & a = frame1Planes[correspondences[i].sourceIndex];
    DetectedPlane const & b = frame2Planes[correspondences[i].targetIndex];

    double const c = 1.0 / std::sqrt(a.rhoCovariance + b.rhoCovariance);
    M_hat.block<1,3>(i,0) = a.plane.normal().transpose().cast<double>() * c;
    d_hat[i] = (a.plane.offset() - b.plane.offset())*c;
  }

  Eigen::JacobiSVD<Eigen::MatrixXd> svd(M_hat, Eigen::ComputeFullU | Eigen::ComputeFullV);
  Eigen::MatrixXd U = svd.matrixU();
  Eigen::Vector3d S = svd.singularValues();
  Eigen::MatrixXd V = svd.matrixV();

  int rank = effectiveRank(U, S, V);

  if(rank > 0)
  {
    // Compute the pseudo inverse of M_hat
    Eigen::MatrixXd M_pinv = Eigen::MatrixXd::Zero(3, correspondences.size());
    for(int i=0; i<rank; ++i)
    {
      auto const & v = V.block<3,1>(0,i);
      auto const & u = U.block(0, i, U.rows(), 1);

      M_pinv +=  v * (1.0/S[i]) * u.transpose();
    }

    // Compute the overlap estimate matrix
    Eigen::Matrix3d M_e = Eigen::Matrix3d::Zero();
    for(int i=rank; i<3; ++i)
      M_e += V.block<3,1>(0,i) * V.block<3,1>(0,i).transpose();

    Eigen::Vector3d overlapEstimate   = Eigen::Vector3d::Zero();
    Eigen::Matrix3d overlapCovariance = Eigen::Matrix3d::Zero();
    if(rank < 3)
    {
      Eigen::Matrix3d R;
      if(rotationEstimate) R = *rotationEstimate;
      else R = pathak::findRotation(correspondences, frame1Planes, frame2Planes).cast<double>();

      overlapEstimate = findTranslationByOverlap(correspondences, frame1Planes, frame2Planes, R, overlapCovariance);
    }

    Eigen::Vector3d const translation = M_pinv * d_hat + M_e * overlapEstimate;
    Eigen::VectorXd const residual = M_hat * translation - d_hat;

    Eigen::Matrix3d covariance = M_pinv * M_pinv.transpose();
    double correctionFactor = residual.dot(residual);
    if((correspondences.size() - rank) > 0)
      correctionFactor /= double(correspondences.size() - rank);
    else
      correctionFactor = std::numeric_limits<double>::max() / 2.0;
    covariance *= correctionFactor;

    covariance += M_e * overlapCovariance * M_e.transpose();

    if(rank_) *rank_ = rank;
    if(covariance_) *covariance_ = covariance;
    if(translationUncertainty_) *translationUncertainty_ = std::abs(covariance.determinant());

    return Eigen::Translation3f(translation.cast<float>());
  }

  return Eigen::Translation3f::Identity();
}





