#pragma once

#include <boost/geometry.hpp>
#include <boost/iterator/iterator_facade.hpp>
#include <nrt/Core/Geometry/Polygon.H>

// Adaptors to allow us to use boost.geometry algorithms directly on nrt data structures

namespace boost
{
  namespace geometry
  {
    namespace traits
    {
      // ######################################################################
      // Point
      // ######################################################################
      template<> struct tag<nrt::Point2D<float>>
      { typedef point_tag type; };

      template<> struct coordinate_type<nrt::Point2D<float>>
      { typedef float type; };

      template<> struct coordinate_system<nrt::Point2D<float>>
      { typedef cs::cartesian type; };

      template<> struct dimension<nrt::Point2D<float>> : boost::mpl::int_<2> {};

      template<>
        struct access<nrt::Point2D<float>, 0>
        {
          static float get(nrt::Point2D<float> const& p)
          { return p.x(); }

          static void set(nrt::Point2D<float>& p, float const& value)
          { p.x() = value; }
        };

      template<>
        struct access<nrt::Point2D<float>, 1>
        {
          static float get(nrt::Point2D<float> const & p)
          { return p.y(); }

          static void set(nrt::Point2D<float>& p, float const & value)
          { p.y() = value; }
        };


      // ######################################################################
      // Polygon
      // ######################################################################
      template<>
        struct tag<std::vector<nrt::Point2D<float>>>
        { typedef ring_tag type; };

      template<class ContainerType>
        class PolygonRingIterator :
          public boost::iterator_facade<PolygonRingIterator<typename ContainerType::value_type>,
          typename ContainerType::value_type, std::random_access_iterator_tag, typename ContainerType::value_type>
      {
        using IteratorType = decltype(std::declval<ContainerType>().begin());
        using PointType = decltype(std::declval<ContainerType>().front());

        public:
          PolygonRingIterator() : pointIndex_(0)
        { }

          explicit PolygonRingIterator(ContainerType & polygon, IteratorType iterator)
            :   polygon_(&polygon), iterator_(iterator), pointIndex_(0)
          { }

          template<class OtherPointType>
            PolygonRingIterator(PolygonRingIterator<OtherPointType> const & other) :
              iterator_(other.getIterator()), pointIndex_(other.getPointIdx())
        { }

          IteratorType getIterator() const {return iterator_;}
          ContainerType * getPolygon() const {return polygon_;}

          bool isEmpty() const { return isEmpty; }
          size_t getPointIdx() const {return pointIndex_;}

          typedef typename boost::iterator_facade<PolygonRingIterator<PointType>, PointType,
                  std::random_access_iterator_tag, PointType>::difference_type difference_type;

        private:
          friend class boost::iterator_core_access;

          void increment()
          {
            ++pointIndex_;
            if (pointIndex_ >= polygon_->size())
            {
              ++iterator_;
              pointIndex_ = 0;
            }
          }

          void decrement()
          {
            if(pointIndex_>0)
            {
              --pointIndex_;
            }
            else
            {
              --iterator_;
              pointIndex_ = polygon_->size();
            }
          }

          void advance(difference_type n)
          {
            difference_type counter = n;

            difference_type maxPointIndex, remainderPointIndex;

            while(counter>0)
            {
              maxPointIndex = polygon_->size(),
                            remainderPointIndex = maxPointIndex - pointIndex_;

              if(counter>remainderPointIndex)
              {
                counter -= remainderPointIndex;
                ++iterator_;
              }
              else // (counter<=remainderPointIndex)
              {
                counter = 0;
                pointIndex_ = remainderPointIndex;
              }
            }
          }

          difference_type distance_to(const PolygonRingIterator& other) const
          {
            IteratorType current_iterator = getIterator();
            IteratorType other_iterator = other.getIterator();

            difference_type count = 0;
            difference_type distance_to_other = std::distance(current_iterator, other_iterator);

            if(distance_to_other < 0)
            {
              count += pointIndex_;

              while(distance_to_other < 0)
              {
                ContainerType const * ls = *other_iterator;
                count -= ls->size();

                ++other_iterator;
                ++distance_to_other;
              }

              assert(other_iterator==current_iterator);
            }
            else if(distance_to_other > 0)
            {
              count -= pointIndex_;

              while(distance_to_other < 0)
              {
                ContainerType const* ls = other.getPolygon();
                count += ls->size();

                ++current_iterator;
                --distance_to_other;
              }

              assert(other_iterator==current_iterator);
            }
            else
            {
              count = pointIndex_ - other.getPointIdx();
            }

            return count;
          }

          bool equal(const PolygonRingIterator& other) const
          {
            return (polygon_ == other.getPolygon()) &&
              (pointIndex_ == other.getPointIdx());
          }

          PointType dereference() const {return (*polygon_)[pointIndex_];}

          ContainerType * polygon_;
          IteratorType iterator_;

          bool empty;
          size_t pointIndex_;
      };

      using RingIterator = PolygonRingIterator<std::vector<nrt::Point2D<float>>>;
      using ConstRingIterator = PolygonRingIterator<const std::vector<nrt::Point2D<float>>>;

      // The required Range functions. These should be defined in the same namespace
      // as Ring.
      inline RingIterator range_begin(std::vector<nrt::Point2D<float>> & r)
      { return RingIterator(r, r.begin()); }

      inline ConstRingIterator range_begin(std::vector<nrt::Point2D<float>> const & r)
      { return ConstRingIterator(r, r.begin()); }

      inline RingIterator range_end(std::vector<nrt::Point2D<float>> & r)
      { return RingIterator(r, r.end()); }

      inline ConstRingIterator range_end(std::vector<nrt::Point2D<float>> const & r)
      { return ConstRingIterator(r, r.end()); }

      // ######################################################################
      // Point
      // ######################################################################
      template<> struct tag<nrt::Point3D<float>>
      { typedef point_tag type; };

      template<> struct coordinate_type<nrt::Point3D<float>>
      { typedef float type; };

      template<> struct coordinate_system<nrt::Point3D<float>>
      { typedef cs::cartesian type; };

      template<> struct dimension<nrt::Point3D<float>> : boost::mpl::int_<3> {};

      template<>
        struct access<nrt::Point3D<float>, 0>
        {
          static float get(nrt::Point3D<float> const& p)
          { return p.x(); }

          static void set(nrt::Point3D<float>& p, float const& value)
          { p.x() = value; }
        };

      template<>
        struct access<nrt::Point3D<float>, 1>
        {
          static float get(nrt::Point3D<float> const & p)
          { return p.y(); }

          static void set(nrt::Point3D<float>& p, float const & value)
          { p.y() = value; }
        };

      template<>
        struct access<nrt::Point3D<float>, 2>
        {
          static float get(nrt::Point3D<float> const & p)
          { return p.z(); }

          static void set(nrt::Point3D<float>& p, float const & value)
          { p.z() = value; }
        };


      // ######################################################################
      // Polygon
      // ######################################################################
      template<>
        struct tag<std::vector<nrt::Point3D<float>>>
        { typedef ring_tag type; };

      template<class ContainerType>
        class PolygonRingIterator3D :
          public boost::iterator_facade<PolygonRingIterator3D<typename ContainerType::value_type>,
          typename ContainerType::value_type, std::random_access_iterator_tag, typename ContainerType::value_type>
      {
        using IteratorType = decltype(std::declval<ContainerType>().begin());
        using PointType = decltype(std::declval<ContainerType>().front());

        public:
          PolygonRingIterator3D() : pointIndex_(0)
        { }

          explicit PolygonRingIterator3D(ContainerType & polygon, IteratorType iterator)
            :   polygon_(&polygon), iterator_(iterator), pointIndex_(0)
          { }

          template<class OtherPointType>
            PolygonRingIterator3D(PolygonRingIterator3D<OtherPointType> const & other) :
              iterator_(other.getIterator()), pointIndex_(other.getPointIdx())
        { }

          IteratorType getIterator() const {return iterator_;}
          ContainerType * getPolygon() const {return polygon_;}

          bool isEmpty() const { return isEmpty; }
          size_t getPointIdx() const {return pointIndex_;}

          typedef typename boost::iterator_facade<PolygonRingIterator3D<PointType>, PointType,
                  std::random_access_iterator_tag, PointType>::difference_type difference_type;

        private:
          friend class boost::iterator_core_access;

          void increment()
          {
            ++pointIndex_;
            if (pointIndex_ >= polygon_->size())
            {
              ++iterator_;
              pointIndex_ = 0;
            }
          }

          void decrement()
          {
            if(pointIndex_>0)
            {
              --pointIndex_;
            }
            else
            {
              --iterator_;
              pointIndex_ = polygon_->size();
            }
          }

          void advance(difference_type n)
          {
            difference_type counter = n;

            difference_type maxPointIndex, remainderPointIndex;

            while(counter>0)
            {
              maxPointIndex = polygon_->size(),
                            remainderPointIndex = maxPointIndex - pointIndex_;

              if(counter>remainderPointIndex)
              {
                counter -= remainderPointIndex;
                ++iterator_;
              }
              else // (counter<=remainderPointIndex)
              {
                counter = 0;
                pointIndex_ = remainderPointIndex;
              }
            }
          }

          difference_type distance_to(const PolygonRingIterator3D& other) const
          {
            IteratorType current_iterator = getIterator();
            IteratorType other_iterator = other.getIterator();

            difference_type count = 0;
            difference_type distance_to_other = std::distance(current_iterator, other_iterator);

            if(distance_to_other < 0)
            {
              count += pointIndex_;

              while(distance_to_other < 0)
              {
                ContainerType const * ls = *other_iterator;
                count -= ls->size();

                ++other_iterator;
                ++distance_to_other;
              }

              assert(other_iterator==current_iterator);
            }
            else if(distance_to_other > 0)
            {
              count -= pointIndex_;

              while(distance_to_other < 0)
              {
                ContainerType const* ls = other.getPolygon();
                count += ls->size();

                ++current_iterator;
                --distance_to_other;
              }

              assert(other_iterator==current_iterator);
            }
            else
            {
              count = pointIndex_ - other.getPointIdx();
            }

            return count;
          }

          bool equal(const PolygonRingIterator3D& other) const
          {
            return (polygon_ == other.getPolygon()) &&
              (pointIndex_ == other.getPointIdx());
          }

          PointType dereference() const {return (*polygon_)[pointIndex_];}

          ContainerType * polygon_;
          IteratorType iterator_;

          bool empty;
          size_t pointIndex_;
      };

      using RingIterator3D = PolygonRingIterator3D<std::vector<nrt::Point3D<float>>>;
      using ConstRingIterator3D = PolygonRingIterator3D<const std::vector<nrt::Point3D<float>>>;

      // The required Range functions. These should be defined in the same namespace
      // as Ring.
      inline RingIterator3D range_begin(std::vector<nrt::Point3D<float>> & r)
      { return RingIterator3D(r, r.begin()); }

      inline ConstRingIterator3D range_begin(std::vector<nrt::Point3D<float>> const & r)
      { return ConstRingIterator3D(r, r.begin()); }

      inline RingIterator3D range_end(std::vector<nrt::Point3D<float>> & r)
      { return RingIterator3D(r, r.end()); }

      inline ConstRingIterator3D range_end(std::vector<nrt::Point3D<float>> const & r)
      { return ConstRingIterator3D(r, r.end()); }

    }
  }
} // namespace boost::geometry::traits

namespace boost
{
    // Specialize metafunctions. We must include the range.hpp header.
    // We must open the 'boost' namespace.
    template <>
    struct range_iterator<nrt::Polygon<float>>
    { typedef geometry::traits::RingIterator type; };

    template<>
    struct range_const_iterator<nrt::Polygon<float>>
    { typedef geometry::traits::ConstRingIterator type; };

} // namespace 'boost'


