#include "PolygonHelpers.H"
#include "PolygonMerge.H"
#include "RefinePlane.H"
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/connected_components.hpp>
#include <boost/geometry.hpp>
#include <boost/geometry/geometries/point_xy.hpp>
#include <boost/geometry/geometries/polygon.hpp>

using namespace std::placeholders;

// ######################################################################
template<size_t numRows>
std::vector<DetectedPlane> mergePolygons(std::vector<DetectedPlane> const & detectedPlanes,
    std::array<std::vector<LiDARGroup>, numRows> const & groups,
    nrt::PointCloud2 const & cloud,
    float const dotProductThresh, float const rhoThresh, float const overlapThresh)
{
  typedef boost::adjacency_list<boost::setS, boost::vecS, boost::undirectedS> Graph;
  Graph planeGraph;
  for(size_t planeIdx1=0; planeIdx1<detectedPlanes.size(); ++planeIdx1)
  {
    Eigen::Hyperplane<float, 3> const & p1 = detectedPlanes[planeIdx1].plane;
    for(size_t planeIdx2=planeIdx1; planeIdx2<detectedPlanes.size(); ++planeIdx2)
    {
      Eigen::Hyperplane<float, 3> const & p2 = detectedPlanes[planeIdx2].plane;

      // TODO: Fix offset (use centroid)
      if(std::abs(p1.offset()-p2.offset()) < rhoThresh && p1.normal().dot(p2.normal()) > dotProductThresh)
        boost::add_edge(planeIdx1, planeIdx2, planeGraph);
    }
  }

  // Find groups whose planes are near each other
  std::vector<size_t> planeComponents(boost::num_vertices(planeGraph));
  size_t numComponents = boost::connected_components(planeGraph, planeComponents.data());
  std::vector<std::vector<DetectedPlane>> componentPlanes(numComponents);
  for(size_t i=0; i<detectedPlanes.size(); ++i)
    componentPlanes[planeComponents[i]].push_back(detectedPlanes[i]);

  std::vector<DetectedPlane> mergedPlanes;
  for(size_t component=0; component<componentPlanes.size(); ++component)
  {
    if(componentPlanes[component].size() == 1)
    {
      DetectedPlane lonelyPlane = componentPlanes[component][0];
      lonelyPlane.area = boost::geometry::area(polygonhelpers::projectPolygon(lonelyPlane.hull, lonelyPlane.plane));
      mergedPlanes.push_back(lonelyPlane);
      continue;
    }

    // Find the average plane for this component
    Eigen::Hyperplane<float, 3> averagePlane = polygonhelpers::findAveragePlane(componentPlanes[component]);

    // Project all of the planes in this component onto the average plane
    std::vector<polygonhelpers::Polygon> projectedPolygons(componentPlanes[component].size());
    std::transform(componentPlanes[component].begin(), componentPlanes[component].end(), projectedPolygons.begin(),
        std::bind(polygonhelpers::projectPlaneHull, _1, averagePlane));

    // Create a graph with each element of the current component as a node.
    // Edges between elements exist if the projected polygons have a non-zero
    // intersection
    Graph componentGraph;
    for(size_t i=0; i<componentPlanes[component].size(); ++i)
    {
      for(size_t j=i; j<componentPlanes[component].size(); ++j)
      {
        std::deque<polygonhelpers::Polygon> intersections;
        try
        {
          boost::geometry::intersection(projectedPolygons[i], projectedPolygons[j], intersections);
        }
        catch(boost::geometry::overlay_invalid_input_exception e)
        {
          std::cerr << "###########################################################" << std::endl;
          std::cerr << "###########################################################" << std::endl;
          std::cerr << "###########################################################" << std::endl;
          std::cerr << "Caught " << e.what() << " in " __FILE__ " on line " << __LINE__ << std::endl;
          std::cerr << "###########################################################" << std::endl;
          std::cerr << "###########################################################" << std::endl;
          std::cerr << "###########################################################" << std::endl;
        }

        if(intersections.size() > 0)
        {
          float const intersectionArea = boost::geometry::area(intersections[0]);
          float const smallestArea = std::min(
              boost::geometry::area(projectedPolygons[i]),
              boost::geometry::area(projectedPolygons[j])
              );

          if(intersectionArea > smallestArea*overlapThresh)
            boost::add_edge(i, j, componentGraph);
        }
      }
    }
    // Find all of the connected components in the graph and assemble them into componentPolygons
    std::vector<size_t> polygonComponents(boost::num_vertices(componentGraph));
    size_t numPolyComponents = boost::connected_components(componentGraph, polygonComponents.data());
    std::vector<std::vector<polygonhelpers::Polygon>> componentPolygons(numPolyComponents);
    std::vector<std::vector<size_t>> componentPolygonIndices(numPolyComponents);
    assert(projectedPolygons.size() == polygonComponents.size());
    for(size_t i=0; i<projectedPolygons.size(); ++i)
    {
      componentPolygons[polygonComponents[i]].push_back(projectedPolygons[i]);
      componentPolygonIndices[polygonComponents[i]].push_back(i);
    }

    // For each subcomponent (overlapping polygon), create a new DetectedPlane
    // as the union of all overlapping polygons.
    for(size_t subcomponent=0; subcomponent<numPolyComponents; ++subcomponent)
    {
      if(componentPolygons[subcomponent].size() == 1)
      {
        DetectedPlane lonelyPlane = componentPlanes[component][componentPolygonIndices[subcomponent][0]];
        lonelyPlane.area = boost::geometry::area(componentPolygons[subcomponent][0]);
        mergedPlanes.push_back(lonelyPlane);
        continue;
      }

      polygonhelpers::Polygon mergedPolygon;
      nrt::Indices mergedIndices;
      for(size_t i=0; i<componentPolygons[subcomponent].size(); ++i)
      {
        std::deque<polygonhelpers::Polygon> unions;
        boost::geometry::union_(mergedPolygon, componentPolygons[subcomponent][i], unions);
        if(unions.size()) boost::geometry::simplify(unions.at(0), mergedPolygon, 0.001);
        mergedIndices.insert(mergedIndices.begin(),
            componentPlanes[component].at(componentPolygonIndices.at(subcomponent).at(i)).indices.begin(),
            componentPlanes[component].at(componentPolygonIndices.at(subcomponent).at(i)).indices.end());
      }

      DetectedPlane mergedPlane;
      mergedPlane.indices = mergedIndices;
      //mergedPlane = refinePlaneFromPoints(mergedPlane, groups, cloud, RefineFrom::Indices);
      mergedPlane.plane = averagePlane;
      mergedPlane.hull = polygonhelpers::reprojectPolygon(mergedPolygon, mergedPlane.plane);
      mergedPlane.area = boost::geometry::area(mergedPolygon);
      mergedPlanes.push_back(mergedPlane);
    }
  }

  return mergedPlanes;
}

template
std::vector<DetectedPlane> mergePolygons<32>(std::vector<DetectedPlane> const & detectedPlanes,
    std::array<std::vector<LiDARGroup>, 32> const & groups,
    nrt::PointCloud2 const & cloud,
    float const dotProductThresh, float const rhoThresh, float const overlapThresh);

template
std::vector<DetectedPlane> mergePolygons<64>(std::vector<DetectedPlane> const & detectedPlanes,
    std::array<std::vector<LiDARGroup>, 64> const & groups,
    nrt::PointCloud2 const & cloud,
    float const dotProductThresh, float const rhoThresh, float const overlapThresh);
