#include "Timing.H"

// ######################################################################
std::map<std::string, timing::Info> & timing::get_info()
{
  static std::map<std::string, timing::Info> timinginfo;
  return timinginfo;
}

// ######################################################################
void timing::start(std::string const & name)
{
  get_info()[name].start = Info::Clock::now();
}

// ######################################################################
void timing::stop(std::string const & name)
{
  Info::Clock::time_point stop = Info::Clock::now();
  assert(get_info().count(name));
  Info & info = get_info()[name];

  Info::Duration duration = std::chrono::duration_cast<Info::Duration>(stop - info.start);

  info.maximum = std::max(duration, info.maximum);

  info.n = info.n + 1;
  Info::Duration delta = duration - info.mean;
  info.mean = info.mean + delta / double(info.n);
  info.m2 = info.m2 + delta.count() * (duration - info.mean);
}

// ######################################################################
timing::Info::Duration timing::mean(std::string const & name)
{
  assert(get_info().count(name));
  return get_info()[name].mean;
}

// ######################################################################
timing::Info::Duration timing::maximum(std::string const & name)
{
  assert(get_info().count(name));
  return get_info()[name].maximum;
}

// ######################################################################
timing::Info::Duration timing::variance(std::string const & name)
{
  assert(get_info().count(name));
  Info & info = get_info()[name];
  return info.m2 / double(info.n - 1);
}

// ######################################################################
void timing::report(std::string const & name)
{
  assert(get_info().count(name));
  std::cout << boost::format("[%|-30|] : %|10t|%07.3fms %|45t|± %2.3f %|55t| (max %07.3fms)") % name % mean(name).count() % sqrt(variance(name).count()) % maximum(name).count();
  std::cout << std::endl;
}

// ######################################################################
void timing::reset(std::string const & name)
{
  assert(get_info().count(name));
  get_info()[name] = timing::Info();
}

// ######################################################################
void timing::reset_all()
{
  for(std::pair<std::string, Info> const & entry : get_info())
    reset(entry.first);
}
