/*! @file
    @author Randolph Voorhies
    @copyright GNU Public License (GPL v3)
    @section License
    @verbatim
    // ////////////////////////////////////////////////////////////////////////
    //              The iLab Neuromorphic Robotics Toolkit (NRT)             //
    // Copyright 2010-2012 by the University of Southern California (USC)    //
    //                          and the iLab at USC.                         //
    //                                                                       //
    //                iLab - University of Southern California               //
    //                Hedco Neurociences Building, Room HNB-10               //
    //                    Los Angeles, Ca 90089-2520 - USA                   //
    //                                                                       //
    //      See http://ilab.usc.edu for information about this project.      //
    // ////////////////////////////////////////////////////////////////////////
    // This file is part of The iLab Neuromorphic Robotics Toolkit.          //
    //                                                                       //
    // The iLab Neuromorphic Robotics Toolkit is free software: you can      //
    // redistribute it and/or modify it under the terms of the GNU General   //
    // Public License as published by the Free Software Foundation, either   //
    // version 3 of the License, or (at your option) any later version.      //
    //                                                                       //
    // The iLab Neuromorphic Robotics Toolkit is distributed in the hope     //
    // that it will be useful, but WITHOUT ANY WARRANTY; without even the    //
    // implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       //
    // PURPOSE.  See the GNU General Public License for more details.        //
    //                                                                       //
    // You should have received a copy of the GNU General Public License     //
    // along with The iLab Neuromorphic Robotics Toolkit.  If not, see       //
    // <http://www.gnu.org/licenses/>.                                       //
    // ////////////////////////////////////////////////////////////////////////
    @endverbatim */

#pragma once

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
#include <Eigen/Core>
#include <DetectedPlane.H>
#pragma GCC diagnostic pop

//! A class to manage the constraint matrix, which tracks how well the ICP system is constrained,
//! and can report on how much the addition of a new point would help the constraints
class PlaneConstraint
{
  public:
    //! Constructor
    /*! @param threshold \sa setThreshold(double threshold) */
    PlaneConstraint(double threshold);

    //! Set a new threshold
    /*! @param threshold The threshold at which a dimension is considered 'fully constrained'. This number
        is the equivalent to the the square-root of the number of vectors in required each dimension.
        For instance, if you need 100 unit vectors in each dimension, then the threshold should be 10. */
    void setThreshold(double threshold);

    //! Get the threshold
    double getThreshold();

    //! Add a constraint to the system in the form of a normal, and the weight.
    /*! @param normal The normal to insert. This is normalized internally.
        @param weight The weight of the constraint, e.g. the number of points this normal represents */
    void addConstraint(Eigen::Vector3f const & normal, double const weight);

    //! Determine how useful a given normal would be to insert into the system
    /*! The higher the score, the more the given normal will help constrain the system.
        @param normal The normal to test
        @returns The score, which varies between 0.0 to 1.0*/
    double getScore(Eigen::Vector3f const & normal);

    //! Get the 3 radii of the constraint ellipsoid.
    /*! To get the equivalent number of normals, the elements of the vector must be squared.
        e.g. `constraint.radii().cwiseProduct(constraint.radii())` */
    Eigen::Vector3d radii();

    //! Get the rotation of the constraint ellipsoid.
    /*! This is generally not needed, but can be useful for plotting the constraint ellipsoid in 3D.*/
    Eigen::Matrix3d rotation();

    //! Get the constraints matrix
    /*! For debugging only */
    Eigen::Matrix3d constraints();

  private:
    //! Compute R_t_ and radii_ with SVD
    /*! This will check and set the dirty_ flag as appropriate */
    void computeSVD();

    double threshold_;      //! The minimum threshold to consider as "constrainted" in any given direction

    Eigen::Matrix3d m_;     //! The constraint matrix

    Eigen::Matrix3d R_t_;     //! The rotation matrix (computed from SVD) transposed
    Eigen::Vector3d radii_; //! The radii (computed from SVD)

    bool dirty_; //! Do we need to recompute SVD on the constraint ellipsoid?
};
