#include "LogFile.h"
#include "RawMessage.h"
#include <assert.h>
#include <stdio.h>
#include <cstring>

//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
namespace DataLog
{

//-----------------------------------------------------------------------------	
LogFile::LogFile() :
    m_r(0), m_w(0)
{
}
//-----------------------------------------------------------------------------
LogFile::~LogFile()
{
    if (m_r)
        fclose(m_r);
    if (m_w)
        fclose(m_w);
}
//-----------------------------------------------------------------------------
bool LogFile::open(const std::string& fileName)
{
    if(m_w || m_r){
        printf("Error: a file is already open");
        return false;
    }
	m_r=fopen64(fileName.c_str(),"rb");
	if (!m_r)
		return false;
	
	
	// Get File Length
	fseeko64(m_r,0,SEEK_END);
	m_fileSize=ftello64(m_r);
	fseeko64(m_r,0,SEEK_SET);
	
	// Check Header (magic number+ version)
	static const unsigned char MAGIC_NUMBER[] = {0xA4, 0x56, 0x45, 0x4C, 0x01, 0x00, 0x01, 0x00};
	unsigned char magicNumber[sizeof(MAGIC_NUMBER)];
	fread( &magicNumber[0], sizeof(MAGIC_NUMBER),1,m_r);
	if (memcmp( &magicNumber[0], &MAGIC_NUMBER[0], sizeof(MAGIC_NUMBER)))
	{
        printf("Error in magic number! \n");
        fclose(m_r);
		m_r=0;
		return false;
	}
	// Read the Index
	unsigned int indexSize = 0;
	fread((char*) &indexSize,sizeof(indexSize),1,m_r);
	m_replayIndex.resize(indexSize);
	fread((char*) &m_replayIndex[0], indexSize*sizeof(long long),1,m_r);
	m_replayLength = -1;
	for( unsigned int i=0; i<indexSize && m_replayIndex[i] != -1; i++ )
		m_replayLength++;
    printf("index length %d replay length %d\n", indexSize, m_replayLength);
	return true;
}
//-----------------------------------------------------------------------------
bool LogFile::seek(int toSec)
{
	if ((toSec<0)||(toSec>m_replayLength))
		return false;
	fseeko64(m_r,m_replayIndex[toSec],SEEK_SET);
	return true;
}
//-----------------------------------------------------------------------------
Message* LogFile::readNextMessage(bool returnRawMessages)
{
	if (!m_r)
		return 0;
	
	FnPtrType handler=0;

	while (!feof(m_r))
	{
		unsigned int msgSize=0;
		fread(&msgSize,sizeof(unsigned int),1,m_r);
		if (msgSize==0xffffffff) // ffffffff indicates eof
		{
            printf("msg size is 0xffffffff\n");
			fclose(m_r);
			m_r=0;
			return 0;
		}
		if (msgSize+ftello64(m_r)+1>m_fileSize) // msg is not fully contained in the log file - eof
		{
            printf("msg is not fully contained in the log file - eof (msg size %d)\n", msgSize);
			fclose(m_r);
			m_r=0;
			return 0;
		}
		unsigned char dummy=0;
		fread(&dummy,sizeof(unsigned char),1,m_r);
		assert(dummy=='1');
		unsigned int msgId=0;
		fread(&msgId,sizeof(unsigned int),1,m_r);

		std::map<unsigned int,FnPtrType>::iterator it=m_msgHandlers.find(msgId);
		if (it==m_msgHandlers.end())
            if(returnRawMessages){
                RawMessage * rawMessage = new RawMessage(m_r, msgSize, msgId);
                return rawMessage;
            }else{
                fseeko64(m_r,msgSize-sizeof(unsigned int)-sizeof(unsigned char),SEEK_CUR); // skip message - no handler for it available
            }
		else
		{
			handler=it->second;
			break;
		}
	}
	if (feof(m_r))
	{
		fclose(m_r);
		m_r=0;
		return 0;
	}


	return handler(m_r);
}
//-----------------------------------------------------------------------------
RawMessage * LogFile::readRawMessage(){
    if (!m_r)
        return 0;

    if (feof(m_r))
    {
        fclose(m_r);
        m_r=0;
        return 0;
    }
    unsigned int msgSize=0;
    fread(&msgSize,sizeof(unsigned int),1,m_r);
    if (msgSize==0xffffffff) // ffffffff indicates eof
    {
        printf("msg size is 0xffffffff\n");
        fclose(m_r);
        m_r=0;
        return 0;
    }
    if (msgSize+ftello64(m_r)+1>m_fileSize) // msg is not fully contained in the log file - eof
    {
        printf("readRawMessage: msg is not fully contained in the log file - eof (msg size %d)\n", msgSize);
        fclose(m_r);
        m_r=0;
        return 0;
    }
    unsigned char dummy=0;
    fread(&dummy,sizeof(unsigned char),1,m_r);
    assert(dummy=='1');
    unsigned int msgId=0;
    fread(&msgId,sizeof(unsigned int),1,m_r);

    RawMessage * rawMessage = new RawMessage(m_r, msgSize, msgId);

    return rawMessage;
}
//-----------------------------------------------------------------------------
bool LogFile::create(const std::string& fileName, unsigned int index_size)
{
    if(m_w || m_r){
        printf("Error: a file is already open");
        return false;
    }
    
    m_w=fopen64(fileName.c_str(),"wb");

    if (!m_w){
        printf("error creating file %s\n", fileName.c_str());
        return false;
    }

    // Set Header (magic number+ version)
    static const unsigned char MAGIC_NUMBER[] = {0xA4, 0x56, 0x45, 0x4C, 0x01, 0x00, 0x01, 0x00};
    fwrite(&MAGIC_NUMBER[0], sizeof(MAGIC_NUMBER),1,m_w);

    // Write the Index
    m_replayIndex.clear();
    m_replayIndex.resize(index_size, -1);
    fwrite(&index_size,sizeof(index_size),1,m_w);

    fwrite(&m_replayIndex[0], index_size*sizeof(long long),1,m_w);
    
    m_currIndexSec = -1;
    return true;
}
//-----------------------------------------------------------------------------
bool LogFile::writeMessage(Message * p_message){
    if(!m_w)
        return false;
    if(!p_message){
        printf("message is 0!");
        return false;
    }

    long long msgPos = ftello64(m_w);
    // write the message
    unsigned int length = p_message->writeMessage(m_w);
    // eventually update the index
    int timeStampSec = static_cast<int>(p_message->getTimestamp()/1000.);
    if(timeStampSec > m_currIndexSec){
        // we have to update the index
        const unsigned int indexPos = 12;
        if(m_currIndexSec < 0) m_currIndexSec = 0;
        for(; m_currIndexSec <= timeStampSec; ++m_currIndexSec){
            // check the bounds
            if(m_currIndexSec >= (int)m_replayIndex.size()){
                printf("size of index not big enough!\n");
                return false;
            }
            // we have to update this index
            m_replayIndex[m_currIndexSec] = msgPos;
            // also write to the file
            fseek(m_w, indexPos+ m_currIndexSec * sizeof(long long), SEEK_SET);
            fwrite(&msgPos, sizeof(long long), 1, m_w);
        }
        fseek(m_w, 0, SEEK_END );
    }
    
    return length != 0xFFFFFFFF;
}

}

