#include "RenderWidget.h"
#include "MainWin.h"
#include "LaserRange2DDataM.h"
#include "LaserRange2DConfigM.h"
#include "IMUStateM.h"
#include "LogFile.h"
#include <QTimer>
#include <QPainter>
#include <QFileDialog>
#include <iostream>
#include <math.h>

//-----------------------------------------------------------------------------
RenderWidget::RenderWidget(MainWin* parent) : QWidget(parent)
{
	m_log=0;
	m_timer=0;
	m_parent=parent;
}
//-----------------------------------------------------------------------------
RenderWidget::~RenderWidget()
{
	delete m_log;
}
//-----------------------------------------------------------------------------
void RenderWidget::paintEvent(QPaintEvent* /*event*/)
{
	QPainter painter(this);
	painter.setPen(Qt::black);
	painter.setBrush(Qt::black);
	painter.drawRect(QRect(0, 0, width() - 1, height() - 1));
	
	// draw laser data
	for (int i=0;i<(int)m_laserPoints.size();i++)
	{
		painter.setPen(QColor(255,255,255));

		int x=-m_laserPoints[i].y()/15+width()/2;
		int y=-m_laserPoints[i].x()/15+height()/2;

		painter.drawPoint(x,y);
	}
}
//-----------------------------------------------------------------------------
void RenderWidget::timeout()
{
	if (!m_log)
		return;

    DataLog::Message* msg=m_log->readNextMessage();
	if (!msg)
	{
		m_timer->stop();
		delete m_log;
		m_log=0;
		return;
	}

	// we read a laser message from the log file
    DataLog::LaserRange2DDataM* laser=dynamic_cast<DataLog::LaserRange2DDataM*>(msg);
	if (laser)
	{
		convertLaserRangesToPoints(laser);
		repaint();
	}

    // we read a laser config message from the log file
    DataLog::LaserRange2DConfigM* config=dynamic_cast<DataLog::LaserRange2DConfigM*>(msg);
    if (config)
    {
        //prevent the msg from being deleted lateron:
        msg = 0;
        // put the new config in
        std::map<std::string, DataLog::LaserRange2DConfigM*>::iterator found = m_laserConfigs.find(config->getSensorType());
        if(found == m_laserConfigs.end()){
            // new entry
            m_laserConfigs[config->getSensorType()] = config;
        }else{
            // replace
            if(found->second) delete found->second;
            found->second = config;
        }
    }

	// we got IMU data from the log file
    DataLog::IMUStateM* imu=dynamic_cast<DataLog::IMUStateM*>(msg);
	if (imu)
	{
		std::cout << "Read IMU Data: " << imu->getQuat()[0] << " " << imu->getQuat()[1] << " " << imu->getQuat()[2] << " " << imu->getQuat()[3] << std::endl;
	}

    if(msg){
        char buffer[256];
        sprintf(buffer,"Current Msg Timestamp: %1.2f",msg->getTimestamp()/1000.0);
        m_parent->setWindowTitle(buffer);
    }

    delete msg;
}
//-----------------------------------------------------------------------------
void RenderWidget::convertLaserRangesToPoints(const DataLog::LaserRange2DDataM* msg)
{

    float startAngle=0.f;
    double scanAngle=0.;
    double deltaAngle=0.;

    std::map<std::string, DataLog::LaserRange2DConfigM*>::iterator found = m_laserConfigs.find(msg->getSensorType());
    if(found != m_laserConfigs.end())
    {
        DataLog::LaserRange2DConfigM * config = found->second;
        // we have a configuration - take it!
        //TODO: do something with the orientation ... for now assume 0, 0, 0
        startAngle = config->getFieldOfViewDeg() /2. / 180.*M_PI * -1;
        scanAngle = config->getFieldOfViewDeg() / 180. * M_PI;
        deltaAngle = scanAngle / static_cast<float>(config->getNumberOfBeams()-1);
    }
    else if (msg->getSensorType()=="Hokuyo_UTM-30LX")
	{
        // use the hokuyo config
        startAngle=-135/180.0*M_PI;
        scanAngle=270/180.0*M_PI;
        deltaAngle=scanAngle/float(int(msg->getRangeData().size())-1);
    }


    if(deltaAngle == 0.)
	{
		m_laserPoints.clear();
		std::cout << "Don't know how to deal with laser " << msg->getSensorType() << std::endl;
        return;
	}
    
    double alpha=startAngle;
    m_laserPoints.resize(msg->getRangeData().size());
    
    for (unsigned i=0; i<msg->getRangeData().size(); i++)
    {
        m_laserPoints[i]=QPoint(cos(alpha)*msg->getRangeData()[i],sin(alpha)*msg->getRangeData()[i]);
        alpha+=deltaAngle;
    }
    
}
//-----------------------------------------------------------------------------
void RenderWidget::open()
{
	QString str=QFileDialog::getOpenFileName(this, tr("Open DataLog"), "", tr("DataLog Files (*.datalog)"));
	if (str=="")
		return;

	delete m_log;
    m_log=new DataLog::LogFile();
	if (!m_log->open(str.toStdString()))
	{
		std::cerr << "Error loading the Replay" << std::endl;
		delete m_log;
		m_log=0;
		return;
	}

    m_log->handleMessage<DataLog::LaserRange2DDataM>();
    m_log->handleMessage<DataLog::LaserRange2DConfigM>();
    m_log->handleMessage<DataLog::IMUStateM>();

	delete m_timer;
	m_timer = new QTimer(this);
	connect(m_timer, SIGNAL(timeout()), this, SLOT(timeout()));
	m_timer->start(10);
}
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
		
