#include "IOUtils.H"

#include <nrt/Core/Util/CompilerUtil.H>

#ifdef USE_MRPT
NRT_BEGIN_UNCHECKED_INCLUDES;
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
#include <mrpt/obs/CRawlog.h>
#include <mrpt/obs/CObservation3DRangeScan.h>
#pragma GCC diagnostic pop
NRT_END_UNCHECKED_INCLUDES;
#endif // USE_MRPT

#include <nrt/Core/Image/Image.H>
#include <nrt/PointCloud2/Common/Transforms.H>
#include <nrt/ImageProc/IO/ImageSource/ImageReaders/ImageReaders.H>

#include <iostream>

#include <boost/format.hpp>

#include "External/PngDistanceImage.hpp"

using namespace nrt;

//namespace
//{
//  // Azimuth offset angle of each laser; 0.0 means horizontal
//  //  I got these values from https://github.com/ros-drivers/velodyne/blob/master/velodyne_pointcloud/params/64e_utexas.yaml
//  //  they should really be PER velodyne, but the Koblenz data format is terrible and you should feel terrible about it
//  static float const LaserAngle[64] = {
//    -30.67, -9.33, -29.33, -8.00, -28.00, -6.66, -26.66, -5.33, -25.33, -4.00, -24.00, -2.67, -22.67,
//    -1.33, -21.33, 0.00, -20.00, 1.33, -18.67, 2.67, -17.33, 4.00, -16.00, 5.33, -14.67, 6.67, -13.33,
//    8.00, -12.00, 9.33, -10.67, 10.67 };
//}

KoblenzReader::KoblenzReader( std::string const & file ) :
  itsCurrCloud( nrt::PointCloud2::create<nrt::PixGray<nrt::byte>>() ),
  itsNextCloud( nrt::PointCloud2::create<nrt::PixGray<nrt::byte>>() ),
  itsLastRotation( 0 ),
  itsFile( file, std::ios::binary )
{
  // Read header
  Header head;
  itsFile.read( reinterpret_cast<char *>( &head ), sizeof( head ) );
  std::cout << "Koblenz header: " << head.magic << " version " << head.major_version << "_" << head.minor_version << std::endl;

  assert( head.magic[0] == 0xA4 );
  assert( head.magic[1] == 'V' );
  assert( head.magic[2] == 'E' );
  assert( head.magic[3] == 'L' );

  // Read index metadata
  IndexHeader indices;
  itsFile.read( reinterpret_cast<char *>( &indices ), sizeof( indices ) );

  std::cout << "Header has " << indices.size << " indices" << std::endl;

  uint64_t firstData;
  std::cout << std::hex;
  itsFile.read( reinterpret_cast<char *>( &firstData ), sizeof( uint64_t ) );

  itsFile.seekg( firstData, std::ios_base::beg );

  ///////////////////////////////////////////////////////////////////////////////////
  static_assert(sizeof(velodyne_koblenz::LaserBlock) == 100, "incorrect size for laser block");
  static_assert(sizeof(velodyne_koblenz::LaserPacket) == 1206, "incorrect size for laser packet");

  // Fill-in our sin/cos tables:
  //for (int i = 0; i < 32; ++i)
  //{ cosVertAngle[i] = cos(LaserAngle[i] * M_PI / 180.0); sinVertAngle[i] = sin(LaserAngle[i] * M_PI / 180.0); }

  for (int i = 0; i < 36000; ++i)
  { cosRotAngle[i] = cos(i * 0.01 * M_PI / 180.0); sinRotAngle[i] = sin(i * 0.01 * M_PI / 180.0); }
}

// ######################################################################
void KoblenzReader::addLaserBlock(velodyne_koblenz::LaserBlock const & lb, nrt::PointCloud2 & cloud,
                                  float const mindist, float const maxdist, bool const extra)
{
  float const cos_rot_angle = cosRotAngle[lb.rotation];
  float const sin_rot_angle = sinRotAngle[lb.rotation];

  int idx = 0;
  for (velodyne_koblenz::LaserData const & ldata : lb.laserdata)
  {
    float const distance = 0.002F * ldata.distance; // convert to meters

    float x, y, z;
    if (distance >= mindist && distance <= maxdist)
    {
      float const cos_vert_angle = cosVertAngle[idx];
      float const sin_vert_angle = sinVertAngle[idx];

      // Compute the distance in the xy plane (w/o accounting for rotation):
      float const xy_distance = distance * cos_vert_angle;

      // Calculate the point coordinates:
      x = xy_distance * cos_rot_angle;
      y = - xy_distance * sin_rot_angle;
      z = distance * sin_vert_angle;
    }
    else
    {
      // Insert a NaN point to signal that the point was no good
      x = std::numeric_limits<nrt::PointCloud2::BaseType>::quiet_NaN();
      y = std::numeric_limits<nrt::PointCloud2::BaseType>::quiet_NaN();
      z = std::numeric_limits<nrt::PointCloud2::BaseType>::quiet_NaN();
    }

    if (extra)
    {
      // Insert (x, y, z, intensity, ExtraLaserData) into our point cloud:
      velodyne_koblenz::ExtraLaserData e; e.rotation = lb.rotation; e.lasernum = idx;
      cloud.insert(nrt::PointCloud2Data<nrt::PixGray<nrt::byte>, velodyne_koblenz::ExtraLaserData>({x, y, z}, nrt::PixGray<nrt::byte>(ldata.intensity), e));
    }
    else
    {
      // Insert (x, y, z, intensity) into our point cloud:
      cloud.insert(nrt::PointCloud2Data<nrt::PixGray<nrt::byte>>({ x, y, z }, nrt::PixGray<nrt::byte>(ldata.intensity)));
    }

    // Switch to next laser in the sequence:
    ++ idx;
  }
}

// ######################################################################
bool KoblenzReader::addLaserPacket( velodyne_koblenz::LaserPacket const & packet, float const mindist, float const maxdist, bool const extra )
{
  bool complete = false;

  // Process the laser samples in the packet:
  for (velodyne_koblenz::LaserBlock const & lb : packet.laserblock)
  {
    if (lb.id != velodyne_koblenz::BLOCK_0_TO_31 || lb.id != velodyne_koblenz::BLOCK_32_TO_63 )
    { NRT_WARNING("Ignoring received packet with unknown firing ID"); continue; }

    // When we wrap around (go back to rotation near 0), we switch to using next from now on, otherwise we use curr:
    if (complete || lb.rotation < itsLastRotation)
    { complete = true; addLaserBlock(lb, itsNextCloud, mindist, maxdist, extra); }
    else addLaserBlock(lb, itsCurrCloud, mindist, maxdist, extra);

    itsLastRotation = lb.rotation;
  }
  return complete;
}

// reads until it gets a point cloud
nrt::Optional<nrt::PointCloud2> KoblenzReader::getCloud( int skipFrames )
{
  int cloudNumber = 0;

  // Keep reading until we get the next point cloud, optionally ignoring up to skipFrames clouds
  while( true )
  {
    if( !itsFile.tellg() )
      return nrt::OptionalEmpty;

    // read and say what next type is
    MessageHeader messageHeader;
    itsFile.read( reinterpret_cast<char *>( &messageHeader ), sizeof( MessageHeader ) );

    // skip to next message
    if( messageHeader.type != MessageType::VelodyneRawDataM || cloudNumber < skipFrames )
    {
      const size_t headerSize = messageHeader.size - sizeof( messageHeader ) + sizeof( messageHeader.size );
      itsFile.seekg( headerSize, std::ios_base::cur );
      continue;
    }

    // otherwise, we have a point cloud!

    // read velodyne header
    VelodyneRawDataMessage veloMessage;
    itsFile.read( reinterpret_cast<char *>( &veloMessage ), sizeof( VelodyneRawDataM ) );

    // read velodyne laser packet
    std::unique_ptr<velodyne_koblenz::LaserPacket[]> lasers( new velodyne_koblenz::LaserPacket[veloMessage.num_packets] );
    itsFile.read( reinterpret_cast<char *>( lasers.get() ), veloMessage.num_packets * sizeof( velodyne_koblenz::LaserPacket ) );

    // Process laser packet
    for( size_t i = 0; i < veloMessage.num_packets; ++i )
      if( addLaserPacket( lasers[i] ) )
        break;
  }
}

// ######################################################################
// ######################################################################
// ######################################################################

VelodyneIO::VelodyneIO( std::string const & file, DataSource source ) : itsDataSource( source ),
  itsReader(), itsFileReader(), itsKoblenz(),
  itsLastFrame( 0 ), itsFinished( false )
{
  switch( itsDataSource )
  {
    case DataSource::nrtlog:
      itsReader.reset( new MessageReader( file, "InputVelodyne" ) );
      break;
#ifdef USE_MRPT
    case DataSource::mrptlog:
      itsRawLog.reset( new mrpt::utils::CFileGZInputStream( file ) );
      break;
#endif // USE_MRPT
    case DataSource::karlsruhe:
      itsFileReader.reset( new std::ifstream( file ) );
      break;
    case DataSource::koblenz:
      itsKoblenz.reset( new KoblenzReader( file ) );
      break;
    case DataSource::kitti:
      itsKittiDirectory = file;
    default:
      break;
  }
}

nrt::Optional<nrt::PointCloud2> VelodyneIO::getCloud()
{
  return getCloud( itsLastFrame + 1 );
}

nrt::Optional<nrt::PointCloud2> VelodyneIO::getCloud( int frame )
{
  nrt::Optional<nrt::PointCloud2> cloud = nrt::OptionalEmpty;

  if( itsFinished || ( frame < itsLastFrame ) )
    return cloud;

  std::cerr << "Getting cloud " << frame << std::endl;

  switch( itsDataSource )
  {
    case DataSource::nrtlog:
      cloud = getCloudNRTLog( frame );
      break;
#ifdef USE_MRPT
    case DataSource::mrptlog:
      cloud = getCloudMRPTRawLog( frame );
      break;
#endif // USE_MRPT
    case DataSource::karlsruhe:
      cloud = getCloudKarlsruhe( frame );
      break;
    case DataSource::koblenz:
      cloud = getCloudKoblenz( frame );
      break;
    case DataSource::kitti:
      cloud = getCloudKitti( frame );
    default:
      break;
  }

  if( cloud )
    itsLastFrame = frame;

  return cloud;
}

nrt::Optional<PointCloud2> VelodyneIO::getCloudNRTLog( int frame )
{
  for( int i = itsLastFrame; i < frame; ++i )
  {
    if(i % 100 == 0) NRT_INFO("Frame " << i << " / " << frame);
    itsReader->next();
  }

  try
  {
    PointCloud2 cloud = itsReader->get();

    //Eigen::Affine3f transform(Eigen::AngleAxisf(-6.0f*M_PI/180.0f, Eigen::Vector3f::UnitX()));

    //transformPointCloudInPlace(cloud, transform);

    cloud.addField<PixRGB<byte>>();
    for(auto p : cloud.range<PixRGB<byte>, PixGray<byte>>()) p.get<PixRGB<byte>>() = PixRGB<byte>(p.get<PixGray<byte>>());
    cloud.removeField<PixGray<byte>>();

    //transformPointCloudInPlace(cloud, Eigen::Affine3f(Eigen::AngleAxisf(-10*M_PI/180.0, Eigen::Vector3f::UnitX())));

    return cloud;
  }
  catch(cereal::Exception const & e)
  {
    itsFinished = true;
    std::cerr << "Finished " << e.what() << std::endl;
    return nrt::OptionalEmpty;
  }
}

#ifdef USE_MRPT
nrt::Optional<nrt::PointCloud2> VelodyneIO::getCloudMRPTRawLog( int frame )
{
  mrpt::obs::CActionCollectionPtr action;
  mrpt::obs::CSensoryFramePtr observations;
  mrpt::obs::CObservationPtr observation;
  size_t entry = itsLastFrame;
  bool valid = false;

  for( int i = itsLastFrame; i < frame; ++i )
    valid = mrpt::obs::CRawlog::getActionObservationPairOrObservation( *itsRawLog, action, observations, observation, entry );

  if( valid && observation )
  {
    mrpt::obs::CObservation3DRangeScanPtr obs = mrpt::obs::CObservation3DRangeScanPtr(observation);
    //auto timestamp = obs->timestamp;

    nrt::PointCloud2 cloud( obs->points3D_x.size() );

    auto & ptsX = obs->points3D_x;
    auto & ptsY = obs->points3D_y;
    auto & ptsZ = obs->points3D_z;

    for( size_t i = 0; i < cloud.size(); ++i )
      cloud[i] = {ptsX[i], ptsY[i], ptsZ[i]};

    std::cerr << "Cloud " << cloud.size() << std::endl;

    return cloud;
  }

  if( !valid )
  {
    itsFinished = true;
    std::cerr << "Finished reading files" << std::endl;
  }

  std::cerr << "Empty" << std::endl;

  return nrt::OptionalEmpty;
}
#endif // USE_MRPT

nrt::Optional<PointCloud2> VelodyneIO::getCloudKarlsruhe( int frame )
{
  std::string currentFileName;

  for( int i = itsLastFrame; i < frame; ++i )
  {
    if(i % 100 == 0) NRT_INFO("Frame " << i << " / " << frame);
    std::getline( *itsFileReader, currentFileName );
  }

  if( itsFileReader->fail() )
  {
    itsFinished = true;
    std::cerr << "Finished reading files" << std::endl;
    return nrt::OptionalEmpty;
  }

  // Yaw angle is linear mapping from image column [0-869] -> [180- -180] in radians
  static auto azimuth = []( size_t col )
  {
    float angle = -(col / 869.0f * 360.0f - 180.0f);
    //angle = angle > 180.0f ? -( angle - 180.0f ) : 180.0f - angle;
    return angle * static_cast<float>( M_PI ) / 180.0f;
  };

  // Vertical angle of each laser (row to vertical angle in degrees)
  static std::array<float, 64> veloAngles = {{
     -1.9367, -1.57397, -1.30476, -0.871566, -0.57881, -0.180617, 0.088762, 0.451829,
      0.80315, 1.20124, 1.49388, 1.83324, 2.20757, 2.54663, 2.87384, 3.23588,
      3.53933, 3.93585, 4.21552, 4.5881, 4.91379, 5.25078, 5.6106, 5.9584,
      6.32889, 6.67575, 6.99904, 7.28731, 7.67877, 8.05803, 8.31047, 8.71141,
      9.02602, 9.57351, 10.0625, 10.4707, 10.9569, 11.599, 12.115, 12.5621,
      13.041, 13.4848, 14.0483, 14.5981, 15.1887, 15.6567, 16.1766, 16.554,
      17.1868, 17.7304, 18.3234, 18.7971, 19.3202, 19.7364, 20.2226, 20.7877,
      21.3181, 21.9355, 22.4376, 22.8566, 23.3224, 23.971, 24.5066, 24.9992 }};

  // Inclination angle in radians for a row
  static auto inclination = [&]( size_t row )
  {
    return ( veloAngles[row] + 90.0f ) * static_cast<float>( M_PI ) / 180.0f;
  };

  try
  {
    // Load PNG file
    PngDistanceImage img( currentFileName );
    auto const width = img.width();
    auto const height = img.height();

    // Create Point Cloud
    PointCloud2 cloud = PointCloud2::create<PixRGB<byte>>( width * height );
    auto cloudIter = cloud.begin<PixRGB<byte>>();

    // algorithm wants data in column major order
    for( size_t col = 0; col < width; ++col )
      for( size_t row = 0; row < height; ++row )
      {
        const float dist = img.getDistance<float>( row, col );


        cloudIter->geometry() = { dist * std::sin( inclination( row ) ) * std::cos( azimuth( col ) ),
                                  dist * std::sin( inclination( row ) ) * std::sin( azimuth( col ) ),
                                  dist * std::cos( inclination( row ) )
                                };

        cloudIter->get<PixRGB<byte>>() = PixRGB<byte>(std::max( dist / PngDistanceImage::maxDst * 255, 64.0)); // scale 0-255

        ++cloudIter;
      }

    return cloud;
  }
  catch(...)
  {
    std::cerr << "Could not open " << currentFileName << std::endl;
    return nrt::OptionalEmpty;
  }
}

nrt::Optional<PointCloud2> VelodyneIO::getCloudKoblenz( int frame )
{
  // pass how many frames we want to skip, 0 or less indicates no skipping
  return itsKoblenz->getCloud( frame - itsLastFrame - 1 );
}

struct PointKitti
{
  float x, y, z, r;

  template <class Archive>
  void serialize( Archive & ar )
  { ar( x, y, z, r ); }
};

nrt::Optional<PointCloud2> VelodyneIO::getCloudKitti( int frame )
{
  // Skip frames if needed
  int actualFrame = itsLastFrame;

  for( int i = actualFrame; i < frame; ++i )
  {
    if(i % 100 == 0) NRT_INFO("Frame " << i << " / " << frame);
    ++actualFrame;
  }

  // Create file name
  auto fmt = boost::format("%|s|/%|06d|.bin") % itsKittiDirectory % actualFrame;
  std::string fileName = fmt.str();

  // Open stream if file is good
  std::ifstream fileStream(fileName, std::ifstream::binary);

  if( !fileStream.is_open() )
  {
    std::cerr << "Finished" << std::endl;
    return nrt::OptionalEmpty;
  }

  // Kitti files are sequential ######.bin up to 6 digits long
  // data is represented as four floating point values:
  // x, y, z, and reflectance. Data is stored in row-major order.
  //
  // Plane algorithm expects column major so we will re-order
  //
  // The number of points can be determined from the file size

  auto fileSize = [&]()
  {
    std::ifstream in(fileName, std::ifstream::ate | std::ifstream::binary);
    return in.tellg();
  };

  const int numPoints = fileSize() / sizeof(PointKitti);
  const int pointsPerRow = numPoints / 64;

  std::vector<std::vector<PointKitti>> points( 64 );
  for( auto & i : points ) i.resize( pointsPerRow );

  cereal::BinaryInputArchive ar( fileStream );
  nrt::PointCloud2 cloud( numPoints );
  cloud.addField<PixRGB<byte>>();

  for( size_t i = 0; i < 64; ++i )
    for( size_t j = 0; j < pointsPerRow; ++j )
      ar( points[i][j] );

  size_t idx = 0;
  for( size_t j = 0; j < pointsPerRow; ++j )
    for( size_t i = 0; i < 64; ++i )
    {
      const auto & pt = points[i][j];
      cloud[idx] = {pt.x, pt.y, pt.z};
      cloud.get<PixRGB<byte>>( idx ).get<PixRGB<byte>>() = PixRGB<byte>(PixGray<byte>(pt.r * 255));
      ++idx;
    }

  return cloud;
}
