#include "DrawingUtils.H"
#include <nrt/ImageProc/Drawing/ColorMapping.H>
#include <nrt/PointCloud2/Common/Centroid.H>
#include <PointCloud/Features/Planes/details/PolygonHelpers.H>
#include <nrt/PointCloud2/Features/Normals.H>

using namespace nrt;

// ######################################################################
nrt::Point3D<float> computeCentroid(std::vector<nrt::Point3D<float>> const & points)
{
  nrt::Point3D<float> c(0, 0, 0);
  for(auto p : points)
    c += p;
  c /= points.size();
  return c;
}

// ######################################################################
graphics::Polygon createPlanePolygon(Eigen::Hyperplane<float, 3> const & plane, PixRGBA<byte> const & color, float size,
    Eigen::Affine3f const & frame, nrt::Duration const & duration)
{
  Eigen::Vector3f normal = -plane.normal();

  int imin = 0;
  for(int i = 0; i<3; ++i)
    if(std::abs(normal[i]) < std::abs(normal[imin]))
      imin = i;

  Eigen::Vector3f v2(0,0,0);
  float dt = normal[imin];

  v2[imin] = 1;
  for(int i=0; i<3; ++i)
    v2[i] -= dt*normal[i];

  Eigen::Vector3f v3 = normal.cross(v2);

  float o = size;

  Eigen::Vector3f p1 = frame*(normal*plane.offset() + o*v2 + o*v3);
  Eigen::Vector3f p2 = frame*(normal*plane.offset() + o*v2 - o*v3);
  Eigen::Vector3f p3 = frame*(normal*plane.offset() - o*v2 - o*v3);
  Eigen::Vector3f p4 = frame*(normal*plane.offset() - o*v2 + o*v3);

  std::vector<nrt::Point3D<float>> points = {
    nrt::Point3D<float>(p1.x(), p1.y(), p1.z()),
    nrt::Point3D<float>(p2.x(), p2.y(), p2.z()),
    nrt::Point3D<float>(p3.x(), p3.y(), p3.z()),
    nrt::Point3D<float>(p4.x(), p4.y(), p4.z()) };

  return graphics::Polygon(points, Eigen::Affine3f::Identity(), duration, color);
}

// ######################################################################
void createHullPolygon(std::vector<std::shared_ptr<nrt::graphics::Shape>> & shapes,
    std::vector<nrt::Point3D<float>> const & hull, PixRGBA<byte> const & color,
    Eigen::Affine3f const & frame, bool drawOutline, bool drawFill, nrt::Duration const & duration, float const lineWidth)
{
  if(!hull.size()) return;

  if(drawFill)
    shapes.push_back(std::make_shared<nrt::graphics::Polygon>(hull, frame, duration, color));

  if(drawOutline)
  {
    PixRGB<byte> lineColor(color*1);

    for(size_t i=1; i<hull.size(); ++i)
    {
      shapes.push_back(std::make_shared<nrt::graphics::Line>(
            frame * Eigen::Vector3f(hull[i-1].x(), hull[i-1].y(), hull[i-1].z()),
            frame * Eigen::Vector3f(hull[i].x(), hull[i].y(), hull[i].z()),
            duration, lineColor, nrt::graphics::Brush(), lineWidth));
    }

    shapes.push_back(std::make_shared<nrt::graphics::Line>(
          frame * Eigen::Vector3f(hull[0].x(), hull[0].y(), hull[0].z()),
          frame * Eigen::Vector3f(hull[hull.size()-1].x(), hull[hull.size()-1].y(), hull[hull.size()-1].z()),
          duration, lineColor, nrt::graphics::Brush(), lineWidth));
  }
}

// ######################################################################
PixRGBA<byte> getNormalColor(Eigen::Vector3f const & n, byte alpha)
{ return PixRGBA<byte>((n.x()+1.0)/2.0*255, (n.y()+1.0)/2.0*255, (n.z()+1.0)/2.0*255, alpha); }

// ######################################################################
void drawPlanes(std::vector<std::shared_ptr<nrt::graphics::Shape>> & shapes,
    std::vector<DetectedPlane> const & planes, Eigen::Isometry3d const & tform, uint8_t const opacity)
{
  for(auto plane : planes)
  {
    Eigen::Isometry3f const t = tform.cast<float>();
    // ubuntu 16.04 gtsam 3.2.1 crashes if rotation() called on float tform
    createHullPolygon(shapes, plane.hull,
        getNormalColor(tform.rotation().cast<float>() * plane.plane.normal(), opacity), t, true, true, nrt::forev());
  }
}

// ######################################################################
void drawSLAMMap(std::vector<std::shared_ptr<nrt::graphics::Shape>> & shapes, PlaneSLAM::RenderedMap const & map)
{
  for(auto const & mapHullPair : map)
  {
    PlaneSLAM::RenderedMapHull mapHull = mapHullPair.second;
    std::string const label(mapHull.symbol);

    auto hull = mapHull.hull;
    uint8_t opacity = 200;
    if(mapHull.plane.normal().dot(-Eigen::Vector3f::UnitZ()) > .75)
      opacity = 64;
    createHullPolygon(shapes, hull, getNormalColor(mapHull.plane.normal(), opacity), Eigen::Affine3f::Identity(), true, true, nrt::forev(), 2.0);

    auto c = computeCentroid(hull);
    Eigen::Affine3f t = Eigen::Affine3f::Identity();
    t.translate(Eigen::Vector3f(c.x(), c.y(), c.z()));
    t.scale(Eigen::Vector3f(0.0005, 0.0005, 0.0005));

    shapes.push_back(std::make_shared<nrt::graphics::Text3D>(label, t, nrt::forev()));
  }
}

// ######################################################################
void drawPath(std::vector<std::shared_ptr<nrt::graphics::Shape>> & shapes, std::vector<Eigen::Isometry3d> const & path)
{
  for(size_t i=0; i<path.size(); ++i)
  {
    nrt::PixRGB<uint8_t> color(nrt::jetPixel<float>(i, 0, path.size()));
    Eigen::Affine3f t(path[i].cast<float>());
    t.scale(0.1);

    shapes.push_back(std::make_shared<nrt::graphics::Box>(t, nrt::forev(), color));
  }
  if(!path.empty())
  {
    Eigen::Affine3f t(path.back().cast<float>());
    t.scale(0.75);
    shapes.push_back(std::make_shared<nrt::graphics::Axes>(t));
  }
}

// ######################################################################
void drawMenu(std::vector<std::shared_ptr<nrt::graphics::Shape>> & shapes,
    std::map<std::string, std::pair<KeySym, bool>> const & drawOptions)
{
  int height = 20;
  int i = 1;
  for(auto option : drawOptions)
    shapes.push_back(
        std::make_shared<graphics::Text2D>(
          option.first + " [" + XKeysymToString(option.second.first) + "] : " + (option.second.second ? "on" : "off"),
          Point2D<int32>(10, height*i++)));
}

// ######################################################################
void drawCorrespondences(std::vector<std::shared_ptr<nrt::graphics::Shape>> & shapes,
    nrt::Correspondences const & correspondences, std::vector<DetectedPlane> const & lastPlanes, std::vector<DetectedPlane> const & planes,
    nrt::PointCloud2 const & lastCloud, nrt::PointCloud2 const & cloud, PlaneSLAM::RenderedMap const & globalMap, Eigen::Isometry3d const & tform, Eigen::Isometry3d const & lastTform)
{
  static PixRGBA<byte> const WHITE(255,255,255,255);
  float const marker_scale = 0.2;

  Eigen::Affine3f drawFrame(tform.cast<float>());
  Eigen::Affine3f lastDrawFrame(lastTform.cast<float>());

  std::set<size_t> matchedPlanes;
  for(nrt::Correspondence const & corr : correspondences)
  {
    matchedPlanes.insert(corr.targetIndex);

    // Grab the centroids of the current and last plane from the correspondence
    DetectedPlane const & p1 = lastPlanes.at(corr.sourceIndex);
    DetectedPlane const & p2 = planes.at(corr.targetIndex);
    PointCloud2::Geometry c1 = nrt::computeCentroid(lastCloud, p1.indices);
    PointCloud2::Geometry c2 = nrt::computeCentroid(cloud, p2.indices);

    // Get the color of the normal
    PixRGBA<byte> color(getNormalColor(p2.plane.normal(), 128));
    //PixRGBA<byte> color(getNormalColor(drawFrame.rotation()*p2.plane.normal(), 128));
    //color.setR(color.r()*1.25); color.setG(color.g()*1.25); color.setB(color.b()*1.25);

    // Draw a box at the last plane
    Eigen::Affine3f t1 = lastDrawFrame;
    t1.translate(Eigen::Vector3f(c1.x(), c1.y(), c1.z()));
    t1.scale(Eigen::Vector3f(marker_scale, marker_scale, marker_scale));
    shapes.push_back(std::make_shared<nrt::graphics::Box>(t1, nrt::forev(), color));

    // Draw a sphere at the current plane
    Eigen::Affine3f t2 = drawFrame;
    t2.translate(Eigen::Vector3f(c2.x(), c2.y(), c2.z()));
    t2.scale(Eigen::Vector3f(marker_scale, marker_scale, marker_scale));
    shapes.push_back(std::make_shared<nrt::graphics::Sphere>(t2, nrt::forev(), color));

    // Draw a line between the box and the sphere
    shapes.push_back(std::make_shared<nrt::graphics::Line>(lastDrawFrame*c1.getVector3Map(), drawFrame*c2.getVector3Map(), nrt::forev(), WHITE));

    // Display the symbol name
    t2.scale(Eigen::Vector3f(0.005, 0.005, 0.005));
    if(p2.symbol)
    {
      //shapes.push_back(std::make_shared<nrt::graphics::Text3D>("[" + std::string(*(p2.symbol)) + "]" + p2.debugMessage, t2, nrt::forev()));
      shapes.push_back(std::make_shared<nrt::graphics::Text3D>("[" + std::string(*(p2.symbol)) + "]", t2, nrt::forev()));
      if(globalMap.count(*p2.symbol))
      {
        // Draw the line between c1 and the global plane
        nrt::Point3D<float> mapCentroid = globalMap.find(*p2.symbol)->second.centroid;
        Eigen::Vector3f mapCentroidV(mapCentroid.x(), mapCentroid.y(), mapCentroid.z());
        shapes.push_back(std::make_shared<nrt::graphics::Line>(drawFrame*c1.getVector3Map(), mapCentroidV, nrt::forev(), WHITE));
      }
    }
    else
    {
      //shapes.push_back(std::make_shared<nrt::graphics::Text3D>("[?] " + std::to_string(p1.pastObservations.size()) + p1.debugMessage, t2, nrt::forev()));
      shapes.push_back(std::make_shared<nrt::graphics::Text3D>("[?] " + std::to_string(p1.pastObservations.size()), t2, nrt::forev()));
    }
  }

  for(size_t i=0; i<planes.size(); ++i)
  {
    if(matchedPlanes.count(i)) continue;

    DetectedPlane const & p = planes[i];

    PixRGBA<byte> color(getNormalColor(p.plane.normal(), 128));
    //PixRGBA<byte> color(getNormalColor(drawFrame.rotation()*p.plane.normal(), 128));
    //color.setR(color.r()*1.25); color.setG(color.g()*1.25); color.setB(color.b()*1.25);

    PointCloud2::Geometry c = nrt::computeCentroid(cloud, p.indices);

    Eigen::Affine3f t = drawFrame;
    t.translate(Eigen::Vector3f(c.x(), c.y(), c.z()));
    t.scale(Eigen::Vector3f(marker_scale, marker_scale, marker_scale));

    if(p.symbol && globalMap.count(*p.symbol) > 0)
    {
      shapes.push_back(std::make_shared<nrt::graphics::Cylinder>(nrt::graphics::Capped::Yes, t, nrt::forev(), color));
      t.translate(Eigen::Vector3f(0, 0, 1.1));
      t.scale(Eigen::Vector3f(0.005, 0.005, 0.005));
      //shapes.push_back(std::make_shared<nrt::graphics::Text3D>("map[" + std::string(*p.symbol) + "]" + p.debugMessage, t, nrt::forev()));
      shapes.push_back(std::make_shared<nrt::graphics::Text3D>("map[" + std::string(*p.symbol) + "]", t, nrt::forev()));

      // Draw the line between the current plane, and the map plane
      nrt::Point3D<float> mapCentroid = globalMap.find(*p.symbol)->second.centroid;
      Eigen::Vector3f mapCentroidV(mapCentroid.x(), mapCentroid.y(), mapCentroid.z());
      shapes.push_back(std::make_shared<nrt::graphics::Line>(drawFrame*c.getVector3Map(), mapCentroidV, nrt::forev(), WHITE));
    }
    //else
    //{
    //  // New plane unmatched
    //  shapes.push_back(std::make_shared<nrt::graphics::Cylinder>(nrt::graphics::Capped::Yes, t, nrt::forev(), color));
    //  t.translate(Eigen::Vector3f(0, 0, 1.1));
    //  t.scale(Eigen::Vector3f(0.005, 0.005, 0.005));
    //  //shapes.push_back(std::make_shared<nrt::graphics::Text3D>("unmatched" + p.debugMessage, t, nrt::forev()));
    //  shapes.push_back(std::make_shared<nrt::graphics::Text3D>("unmatched", t, nrt::forev()));
    //}
  }
}

// ######################################################################
void drawCurrentCloud(std::vector<std::shared_ptr<nrt::graphics::Shape>> & shapes, Eigen::Isometry3d const & tform, PointCloud2 cloud)
{
  if(cloud.size() == 0) return;

  if(cloud.hasField<PixRGB<uint8_t>>())
  {
    for(auto p : cloud.range<PixRGB<uint8_t>>())
    {
      auto v = p.get<PixRGB<uint8_t>>().r();
      p.get<PixRGB<uint8_t>>() = {v, v, v};
    }
  }
  else
  {
    cloud.addField<PixRGB<uint8_t>>();
    for(auto p : cloud.range<PixRGB<uint8_t>>())
    {
      p.get<PixRGB<uint8_t>>() = {255, 255, 255};
    }
  }

  auto cloudGraphics = std::make_shared<graphics::PointCloud>(cloud, 1.0, tform.cast<float>());
  cloudGraphics->setDisplayType(graphics::PointCloud::Cube);
  cloudGraphics->setPointSize(0.03);
  shapes.push_back(cloudGraphics);
}

// ######################################################################
void drawNormals(std::vector<std::shared_ptr<nrt::graphics::Shape>> & shapes, Eigen::Isometry3d const & tform, PointCloud2 cloud,
    bool scaleByCurvature)
{
  if(cloud.size() == 0) return;
  shapes.push_back(std::make_shared<graphics::PointCloudNormals>(cloud, scaleByCurvature, 0.4F, tform.cast<float>()));
}

// ######################################################################
void drawPICPPoints(std::vector<std::shared_ptr<nrt::graphics::Shape>> & shapes, Eigen::Isometry3d const & tform, PointCloud2 cloud, nrt::Indices indices,
    nrt::PixRGB<uint8_t> color)
{
  if(cloud.size() == 0 || indices.size() == 0) return;

  nrt::PointCloud2 indicesCloud(indices.size());
  for(size_t i=0; i<indices.size(); ++i)
    indicesCloud[i] = cloud[indices[i]];

  indicesCloud.addField<PixRGB<uint8_t>>();
  for(auto p : indicesCloud.range<PixRGB<uint8_t>>())
    p.get<PixRGB<uint8_t>>() = color;

  auto indicesCloudGraphics = std::make_shared<graphics::PointCloud>(indicesCloud, 1.0, tform.cast<float>());
  indicesCloudGraphics->setDisplayType(graphics::PointCloud::Cube);
  indicesCloudGraphics->setPointSize(0.08);
  shapes.push_back(indicesCloudGraphics);
}

// ######################################################################
template <size_t numRows>
void drawBreaksGroups(std::vector<std::shared_ptr<nrt::graphics::Shape>> & shapes, Eigen::Isometry3d const & tform, PointCloud2 const cloud,
    std::array<std::vector<LiDARGroup>, numRows> const & groups, bool draw_breaks, bool draw_groups )
{
  if(cloud.size() == 0) return;

  if( !draw_breaks && !draw_groups ) return;

  nrt::PointCloud2 breaksCloud = nrt::PointCloud2::create<nrt::PixRGB<uint8_t>>();
  nrt::PointCloud2 groupsCloud = nrt::PointCloud2::create<nrt::PixRGB<uint8_t>>();
  std::srand(0);

  for( auto const & laser : groups )
  {
    for( auto const & group : laser )
    {
      nrt::PixRGB<nrt::byte> color(rand()/float(RAND_MAX)*256, rand()/float(RAND_MAX)*256, rand()/float(RAND_MAX)*256);

      auto const & p1 = group.endPoints.first;
      auto const & p2 = group.endPoints.second;

      breaksCloud.insert<nrt::PixRGB<uint8_t>>( {{p1.x(), p1.y(), p1.z()}, color} );
      breaksCloud.insert<nrt::PixRGB<uint8_t>>( {{p2.x(), p2.y(), p2.z()}, color} );

      for( size_t idx : group.indices )
        groupsCloud.insert<nrt::PixRGB<uint8_t>>({cloud[idx], color});
    }
  }

  if( draw_breaks )
  {
    auto cloudGraphics = std::make_shared<nrt::graphics::PointCloud>(breaksCloud, 1.0, tform.cast<float>());
    cloudGraphics->setDisplayType(graphics::PointCloud::Cube);
    cloudGraphics->setPointSize(0.06);
    shapes.push_back(cloudGraphics);
  }

  if( draw_groups )
  {
    auto cloudGraphics = std::make_shared<nrt::graphics::PointCloud>(groupsCloud, 1.0, tform.cast<float>());
    cloudGraphics->setDisplayType(graphics::PointCloud::Cube);
    cloudGraphics->setPointSize(0.04);
    shapes.push_back(cloudGraphics);
  }
}
template
void drawBreaksGroups<32>(std::vector<std::shared_ptr<nrt::graphics::Shape>> & shapes, Eigen::Isometry3d const & tform, PointCloud2 const cloud,
    std::array<std::vector<LiDARGroup>, 32> const & groups, bool draw_breaks, bool draw_groups);
template
void drawBreaksGroups<64>(std::vector<std::shared_ptr<nrt::graphics::Shape>> & shapes, Eigen::Isometry3d const & tform, PointCloud2 const cloud,
    std::array<std::vector<LiDARGroup>, 64> const & groups, bool draw_breaks, bool draw_groups);

// ######################################################################
void drawIntegratedCloud(std::vector<std::shared_ptr<nrt::graphics::Shape>> & shapes,
    std::vector<Eigen::Isometry3d> const & transformHistory, std::vector<PointCloud2> const & cloudHistory)
{
  for(size_t i=0; i<cloudHistory.size(); ++i)
    shapes.push_back(
        std::make_shared<graphics::PointCloud>(cloudHistory[i], 1.0, transformHistory[i].cast<float>()));
}

// ######################################################################
struct JetCache
{
  JetCache(float min_height, float max_height, size_t ncolors = 256) :
    min_height_(min_height), max_height_(max_height), ncolors_(ncolors),
    normalizer_(ncolors_ / (max_height_ - min_height_))
  {
    colors_.resize(ncolors_);
    for(size_t i=0; i<ncolors; ++i)
      colors_.at(i) = nrt::PixRGB<uint8_t>(nrt::jetPixel<float>(i, 0, ncolors-1));
  }

  JetCache & operator=(JetCache &&) = default;

  nrt::PixRGB<uint8_t> const & operator()(float z)
  {
    if(z != z) return invalid_;
    else if(z < min_height_) return colors_.front();
    else if(z >= max_height_) return colors_.back();
    return colors_.at((z - min_height_)*normalizer_);
  }

  float min_height_;
  float max_height_;
  size_t ncolors_;
  float normalizer_;
  std::vector<nrt::PixRGB<uint8_t>> colors_;
  nrt::PixRGB<uint8_t> invalid_ = nrt::PixRGB<uint8_t>(255,255,255);
};

// ######################################################################
void drawIntegratedJetCloud(std::vector<std::shared_ptr<nrt::graphics::Shape>> & shapes,
    std::vector<Eigen::Isometry3d> const & transformHistory, std::vector<PointCloud2> const & cloudHistory, float const minHeight, float const maxHeight)
{
  static std::vector<std::shared_ptr<graphics::PointCloud>> cloudGraphics;
  static JetCache jet(minHeight, maxHeight);

  if(minHeight != jet.min_height_ || maxHeight != jet.max_height_)
    jet = JetCache(minHeight, maxHeight);

  if(cloudHistory.size() != cloudGraphics.size())
  {
    cloudGraphics.clear();

    for(size_t i=0; i<cloudHistory.size(); ++i)
    {
      auto t = transformHistory[i].cast<float>();
      PointCloud2 cloud = cloudHistory[i];

      if(cloud.hasField<PixRGB<uint8_t>>())
      {
        for(auto p : cloud.range<PixRGB<uint8_t>>())
        {
          p.get<nrt::PixRGB<uint8_t>>() = jet((t * p.geometry().getVector3Map()).z())
            * (p.get<nrt::PixRGB<uint8_t>>().r()/255.0);
        }
      }
      else
      {
        cloud.addField<PixRGB<uint8_t>>();
        for(auto p : cloud.range<PixRGB<uint8_t>>())
        {
          p.get<nrt::PixRGB<uint8_t>>() = jet((t * p.geometry().getVector3Map()).z());
        }
      }

      cloudGraphics.push_back(std::make_shared<graphics::PointCloud>(cloud, 1.0, transformHistory[i].cast<float>()));
    }
  }

  for(auto g : cloudGraphics) shapes.push_back(g);
}

// ######################################################################
void drawPlanarity(std::vector<std::shared_ptr<nrt::graphics::Shape>> & shapes, Eigen::Isometry3d const & tform, nrt::PointCloud2 cloud)
{
  static JetCache jet(0, 1);

  if(cloud.size() == 0) return;

  cloud.addField<PixRGB<uint8_t>>();
  for(auto p : cloud.range<PixRGB<uint8_t>, nrt::PointNormal>())
    p.get<PixRGB<uint8_t>>() = jet(1.0 - p.get<nrt::PointNormal>()[3]);

  auto cloudGraphics = std::make_shared<graphics::PointCloud>(cloud, 1.0, tform.cast<float>());
  cloudGraphics->setDisplayType(graphics::PointCloud::Cube);
  cloudGraphics->setPointSize(0.03);
  shapes.push_back(cloudGraphics);
}

