#pragma once
#include <fstream>
#include <iostream>
#include <gtsam/nonlinear/Values.h>
#include <gtsam/nonlinear/Symbol.h>
#include <gtsam/slam/BetweenFactor.h>
#include <gtsam/nonlinear/NonlinearFactorGraph.h>

// ######################################################################
void saveGraphGEXF(std::string const & filename, gtsam::NonlinearFactorGraph const & graph, std::map<gtsam::Symbol, Eigen::Hyperplane<float, 3>> const & map)
{
  std::cout << "SAVING GRAPH TO GEXF" << std::endl;
  gtsam::KeyFormatter formatter = gtsam::DefaultKeyFormatter;
  std::ofstream f(filename);

  f << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
  f << "<gexf xmlns=\"http://www.gexf.net/1.2draft\" xmlns:viz=\"http://www.gexf.net/1.2draft/viz\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.gexf.net/1.2draft http://www.gexf.net/1.2draft/gexf.xsd\" version=\"1.2\">\n";
  f << "    <graph mode=\"static\" defaultedgetype=\"directed\">\n";

  f << "        <attributes class=\"node\">\n";
  f << "            <attribute id=\"0\" title=\"plane\" type=\"string\"/>\n";
  f << "            <attribute id=\"1\" title=\"position\" type=\"string\"/>\n";
  f << "        </attributes>\n";

  f << "        <nodes>\n";
  for(gtsam::Key key : graph.keys())
  {
    f << "            <node id=\"" << key << "\" label=\"" << formatter(key) << "\">\n";
    if(formatter(key)[0] == 'x')
    {
      f << "              <viz:color r=\"255\" g=\"128\" b=\"128\" a=\"1.0\"/>\n";
      f << "              <viz:position x=\"" + std::to_string(gtsam::Symbol(key).index()*50) + "\" y=\"0.0\" z=\"0.0\"/>\n";
    }
    else if(formatter(key)[0] == 'z')
    {

      f << "              <viz:color r=\"128\" g=\"128\" b=\"255\" a=\"1.0\"/>\n";
      f << "              <viz:position x=\"" + std::to_string(gtsam::Symbol(key).index()*50) + "\" y=\"100.0\" z=\"0.0\"/>\n";

      auto planeToString = [](Eigen::Hyperplane<float, 3> const & p) {
        return std::to_string(p.offset()) + " : (" + std::to_string(p.normal().x()) + ", " + std::to_string(p.normal().y()) + ", " + std::to_string(p.normal().z()) + ")";
      };
      gtsam::Symbol symbol(key);
      auto it = map.find(symbol);
      if(it != map.end())
      {
        f << "              <attvalues>\n";
        f << "                  <attvalue for=\"0\" value=\""+planeToString(it->second)+"\"/>\n";
        f << "              </attvalues>\n";
      }
    }
    f << "            </node>\n";
  }
  f << "        </nodes>\n";

  // Grab all of the unique factors
  std::set<std::pair<gtsam::Key, gtsam::Key>> edges;
  for(auto const & factor : graph)
  {
    for(auto i=factor->begin(); i<factor->end(); ++i)
      for(auto j=i+1; j<factor->end(); ++j)
        edges.insert(std::make_pair(*i, *j));
  }

  f << "        <edges>\n";
  size_t edgeId = 0;
  for(auto p : edges)
  {
    f << "            <edge id=\"" << edgeId++ << "\" source=\"" << p.first << "\" target=\"" << p.second << "\"" << ">\n";
    f << "            </edge>\n";

  }
  f << "        </edges>\n";

  f << "    </graph>\n";
  f << "</gexf>\n";

}


// ######################################################################
void saveGraphDot(std::string const & filename, gtsam::NonlinearFactorGraph const & graph)
{
  gtsam::KeyFormatter formatter = gtsam::DefaultKeyFormatter;
  std::ofstream f(filename);
  f << "graph {\n";
  for(gtsam::Key key : graph.keys())
  {
    std::string keyString = formatter(key);
    if(keyString[0] == 'x')
    {
      f << keyString << "[shape=box, color=\"1 .8 .8\"];" << std::endl;
    }
    else if(keyString[0] == 'z')
      f << keyString << "[shape=circle, color=\".8 .8 1\"];" << std::endl;
  }

  // Grab all of the factors
  std::set<std::pair<gtsam::Key, gtsam::Key>> edges;
  for(auto const & factor : graph)
  {
    for(auto i=factor->begin(); i<factor->end(); ++i)
      for(auto j=i+1; j<factor->end(); ++j)
        edges.insert(std::make_pair(*i, *j));
  }
  for(auto p : edges)
  {
    std::string left = formatter(p.first);
    std::string right = formatter(p.second);
    std::string colorstring = right[0] == 'x' ? " [color=red, penwidth=3]" : "";
    f << left << " -> " << right << colorstring << ";" << std::endl;
  }

  f << "}";
}
