module Geometry

export skew, rodrigues, inv_rodrigues, make_transform, rotx, roty, rotz

######################################################################
function skew(x)
    return [   0  -x[3]  x[2];
    x[3]    0  -x[1];
    -x[2]  x[1]    0];
end

######################################################################
function rodrigues(v)
    omega = [v]

    if(size(omega) == (1,3))
        omega = omega'
    elseif(length(omega) != 3)
        error("v must be 1x3 or 3x1")
    end

    theta = norm(omega)

    # If theta is tiny, we will have numerical issues
    theta2            = 0.0
    sinthetatheta     = 0.0
    onecosthetatheta2 = 0.0
    if(theta < sqrt(eps())*1e2)
        theta2            = omega' * omega
        sinthetatheta     = 1 - theta2 / 6.0
        onecosthetatheta2 = 1 - theta2 / 24.0
    else
        theta2            = theta * theta
        sinthetatheta     = sin(theta) / theta
        onecosthetatheta2 = (1 - cos(theta)) / theta2
    end

    omegav = skew(omega)

    R = eye(3) + omegav * sinthetatheta + omegav * omegav * onecosthetatheta2 

    return R;
end

######################################################################
function inv_rodrigues(R)
    trc = trace(R)
    trc2 = (trc-1.0) / 2.0
    sinacostrc2 = sqrt(1.0 - trc2*trc2)

    s = [R[3,2] - R[2,3];
    R[1,3] - R[3,1];
    R[2,1] - R[1,2]]

    if(1 - trc2 * trc2 >= eps())
        theta = acos(trc2)
        thetaf = theta / (2.0*sin(theta));
    else
        theta = real(acos(trc2))
        thetaf = 0.5 / (1.0 - theta/6.0)
    end

    omega = thetaf * s

    return omega
end

######################################################################
function make_transform(T_vec)
    if(length(T_vec) != 6)
        error("make_transform takes a 6 element vector: [r0 r1 r2 t0 t1 t2]")
    end

    R = rodrigues(T_vec[1:3]);
    t = T_vec[4:6];

    return [R t; [0 0 0 1]];
end

######################################################################
function rotx(t)
    return rodrigues([1 0 0]*t)
end

######################################################################
function roty(t)
    return rodrigues([0 1 0]*t)
end

######################################################################
function rotz(t)
    return rodrigues([0 0 1]*t)
end
end
