/*! @file MessageLogging/MessageLoader/MessageLoaderModule.H */

#ifndef MESSAGELOGGING_MESSAGELOADER_MESSAGELOADERMODULE_H
#define MESSAGELOGGING_MESSAGELOADER_MESSAGELOADERMODULE_H

#include <nrt/Core/Blackboard/Module.H>
#include <nrt/PointCloud2/PointCloud2.H>
#include <nrt/Core/Design/BoundedBuffer.H>
#include "../../../../Algorithms/Testing/MessageLogging/MessageSaverData.H"
#include <nrt/Core/Image/GenericImage.H>
#include "../../../../Messages/Robotics/GaussianPoseMessage.H"
#include "../../../../Messages/Drivers/VelodyneIMUMessage.H"

// namespace for ports and parameters of MessageLoaderModule
namespace messageloadermodule
{
  //! Parameter category definition
  nrt::ParameterCategory const ParamCateg("MessageLoader Parameters");

  //! Parameter definition for file name
  nrt::ParameterDef<std::string> const
  ParamDefFilename("filename", "File name to load messages from", "messages.nrtlog", ParamCateg);

  //! Parameter definition for loop
  nrt::ParameterDef<bool> const
  ParamDefLoop("loop", "Loop over the data file if true", false, ParamCateg);

  //! Parameter for use trigger
  nrt::ParameterDef<bool> const
  ParamDefTrigger("use-trigger", "If true, pause after each message is posted and until a trigger message "
                  "is received on InputTrigger", false, ParamCateg);

  //! Parameter for use order post
  nrt::ParameterDef<bool> const
  ParamDefOrderPost("use-order-post", "Post the messages in the order they appear in the file. Otherwise, insert them into "
                  "a queue and post the message using its timestamp", false, ParamCateg);

  //! Checker port for trigger
  NRT_DECLARE_MESSAGECHECKER_PORT(InputTrigger, nrt::TriggerMessage, "Input Optional Trigger");
}

namespace messageloadersub
{
  //! Poster port for velodyne
  NRT_DECLARE_MESSAGEPOSTER_PORT(OutputVelodyne, nrt::Message<nrt::PointCloud2>, void, "Output Velodyne");

  //! Poster port for xtion
  NRT_DECLARE_MESSAGEPOSTER_PORT(OutputXtion, nrt::Message<nrt::PointCloud2>, void, "Output Xtion");

  //! Poster port for xtion image
  NRT_DECLARE_MESSAGEPOSTER_PORT(OutputXtionIm, nrt::Message<nrt::GenericImage>, void, "Output Xtion image");

  //! Poster port for pose
  NRT_DECLARE_MESSAGEPOSTER_PORT(OutputPose, GaussianPoseMessage, void, "Output pose");

  //! Poster port for imu
  NRT_DECLARE_MESSAGEPOSTER_PORT(OutputIMU, VelodyneIMUMessage, void, "Output IMU data");

  //! Poster port for bumblebee cloud
  NRT_DECLARE_MESSAGEPOSTER_PORT(OutputBBcloud, nrt::Message<nrt::PointCloud2>, void, "Output Bumblebee cloud");
  
  //! Poster port for bumblebee disparity image
  NRT_DECLARE_MESSAGEPOSTER_PORT(OutputBBdisparity, nrt::Message<nrt::GenericImage>, void, "Output Bumblebee disparity");
  
  //! Poster port for bumblebee left image
  NRT_DECLARE_MESSAGEPOSTER_PORT(OutputBBleft, nrt::Message<nrt::GenericImage>, void, "Output Bumblebee left image");

  //! Poster port for bumblebee right image
  NRT_DECLARE_MESSAGEPOSTER_PORT(OutputBBright, nrt::Message<nrt::GenericImage>, void, "Output Bumblebee right image");
  
  //! Poster port for bumblebee depth
  NRT_DECLARE_MESSAGEPOSTER_PORT(OutputBBdepth, nrt::Message<nrt::GenericImage>, void, "Output Bumblebee depth image");
  
  //! Poster port for microstrain imu
  NRT_DECLARE_MESSAGEPOSTER_PORT(OutputMicrostrainIMU, GaussianPoseMessage, void, "Output Microstrain IMU");
}

template <class Port> class MessageLoaderSub;

//! Load a log file from MessageSaverModule and post the messages
/*! Each port runs in its own thread. We try to respect the original timing of the messages.

    @author Laurent Itti
    @keywords Message, log, trace */
class MessageLoaderModule : public nrt::Module,
                            public nrt::MessageChecker<messageloadermodule::InputTrigger>
{
  public:
    //! Constructor
    MessageLoaderModule(std::string const & instanceName = "");

    //! Virtual destructor for safe inheritance
    virtual ~MessageLoaderModule();

  private:
    void run();
    template <class Port> friend class MessageLoaderSub;

    nrt::Parameter<std::string> itsFilename;
    nrt::Parameter<bool> itsLoop;
    nrt::Parameter<bool> itsUseTrigger;
    nrt::Parameter<bool> itsUseOrderPost;

    std::shared_ptr<MessageLoaderSub<messageloadersub::OutputVelodyne> > itsVelodyne;
    std::shared_ptr<MessageLoaderSub<messageloadersub::OutputXtion> > itsXtion;
    std::shared_ptr<MessageLoaderSub<messageloadersub::OutputXtionIm> > itsXtionIm;
    std::shared_ptr<MessageLoaderSub<messageloadersub::OutputPose> > itsPose;
    std::shared_ptr<MessageLoaderSub<messageloadersub::OutputIMU> > itsIMU;
    std::shared_ptr<MessageLoaderSub<messageloadersub::OutputBBcloud> > itsBBcloud;
    std::shared_ptr<MessageLoaderSub<messageloadersub::OutputBBdisparity> > itsBBdisparity;
    std::shared_ptr<MessageLoaderSub<messageloadersub::OutputBBleft> > itsBBleft;
    std::shared_ptr<MessageLoaderSub<messageloadersub::OutputBBright> > itsBBright;
    std::shared_ptr<MessageLoaderSub<messageloadersub::OutputBBdepth> > itsBBdepth;
    std::shared_ptr<MessageLoaderSub<messageloadersub::OutputMicrostrainIMU> > itsMicrostrainIMU;
};


//! Helper MessageLoader for one port
template <class Port>
class MessageLoaderSub : public nrt::Module, public nrt::MessagePoster<Port>
{
  public:
    //! Constructor
    MessageLoaderSub(std::string const & instanceName = "");

    //! Virtual destructor for safe inheritance
    virtual ~MessageLoaderSub();

    //! Immediately post a message
    void dopost(std::shared_ptr<MessageSaverData> data);

  protected:
    void run();
    void postStop();

  private:
    friend class MessageLoaderModule;
    nrt::BoundedBuffer<std::shared_ptr<MessageSaverData> > itsQ;
};


#endif // MESSAGELOGGING_MESSAGELOADER_MESSAGELOADERMODULE_H
