#pragma once
#include <Eigen/Core>

class VonMisesFischer
{
    using Vector = Eigen::Vector3f;

  public:
    //! Constructor
    VonMisesFischer() :
      dirty_(true),
      n_(0),
      sum_(Vector::Zero())
    { }


    //! Add a vector to the distribution
    inline void add(Vector const & v)
    {
      sum_ += v;
      ++n_;
      dirty_ = true;
    }

    //! Get the score (from 0-1) of a vector given the distribution
    inline Vector::Scalar evaluate(Vector const & v)
    {
      if( dirty_ )
        compute();
      return std::exp(k_ * mean_.dot(v)) / max_;
    }

  private:
    //! Compute the mean and approximate the concentration (k) of the distribution
    /*! This uses the formulation of Sra 2011 "A short note on parameter approximation for von Mises-Fisher
        distributions: And a fast implementation of I s (x)" */
    inline void compute()
    {
      auto R = sum_.norm() / n_;
      k_ = R * (3.0-R*R) / (1.0 - R*R);

      mean_ = sum_.normalized();
      dirty_ = false;

      max_ = 1.0;
      max_ = evaluate(mean_);
    }

    bool dirty_;  //! Does the distribution need to be recomputed
    size_t n_;    //! Number of vectors added to the distribution
    Vector sum_;  //! Running sum of all vectors added to the distribution
    Vector mean_; //! The computed mean of the distrubution

    Vector::Scalar k_;   //! The estimated k value of the distribution
    Vector::Scalar max_; //! The maximum value of the distribution
};

