#include <nrt/External/nanoflann/nanoflann.hpp>
#include <nrt/PointCloud2/PointCloud2.H>

class NanoFlann
{
  // ######################################################################
  struct PointCloudAdaptor
  {
    typedef nrt::PointCloud2::BaseType coord_t;

    // ######################################################################
    PointCloudAdaptor(nrt::PointCloud2 const & cloud) : itsCloud(cloud)
    { }

    // ######################################################################
    size_t kdtree_get_point_count() const { return itsCloud.size(); }

    // ######################################################################
    coord_t kdtree_distance(coord_t const * p1, size_t const idx_p2, size_t) const
    {
      auto const & p2 = itsCloud[idx_p2];
      coord_t const dx = p1[0] - p2.x();
      coord_t const dy = p1[1] - p2.y();
      coord_t const dz = p1[2] - p2.z();

      return dx*dx + dy*dy + dz*dz;
    }

    // ######################################################################
    coord_t kdtree_get_pt(size_t const idx, int dim) const
    {
      switch(dim)
      {
        case 0: return itsCloud[idx].x();
        case 1: return itsCloud[idx].y();
        case 2: return itsCloud[idx].z();
        default: return std::numeric_limits<coord_t>::signaling_NaN();
      }
    }

    // ######################################################################
    template<class BBox>
      bool kdtree_get_bbox(BBox &) const { return false; }

    nrt::PointCloud2 const itsCloud;
  };


  public:
    using BaseType = nrt::PointCloud2::BaseType;
    using Geometry = nrt::PointCloud2::Geometry;

    // ######################################################################
    NanoFlann(nrt::PointCloud2 const input, float const epsilon = 0, size_t const maxLeaf = 10) :
      itsAdaptor(input),
      itsTree(3, itsAdaptor, nanoflann::KDTreeSingleIndexAdaptorParams(maxLeaf)),
      itsEpsilon(epsilon)
  { itsTree.buildIndex(); }

    ~NanoFlann() {}

    // ######################################################################
    std::vector<std::pair<size_t, BaseType>> knn(Geometry const query, size_t const k) const
    {
      if(itsAdaptor.itsCloud.size() < k)
        throw std::runtime_error("Could not find " + std::to_string(k)
            + " nearest neighbors in cloud of size " + std::to_string(itsAdaptor.itsCloud.size()));

      std::vector<size_t> indices(k);
      std::vector<BaseType> square_distances(k);

      itsTree.knnSearch(&query.x(), k, &indices[0], &square_distances[0]);

      std::vector<std::pair<size_t, BaseType>> results(k);
      auto results_it = results.begin();
      auto results_end = results.end();
      auto indices_it = indices.begin();
      auto dists_it = square_distances.begin();

      for(; results_it != results_end; ++results_it, ++indices_it, ++dists_it)
        *results_it = std::make_pair(*indices_it, *dists_it);

      return results;
    }

    // ######################################################################
    std::pair<size_t, BaseType> nn(Geometry const query) const
    {
      if(itsAdaptor.itsCloud.size() == 0)
        throw std::runtime_error("Could not find nearest neighbor in cloud of size " +
            std::to_string(itsAdaptor.itsCloud.size()));

      std::pair<size_t, BaseType> ret;
      itsTree.knnSearch(&query.x(), 1, &ret.first, &ret.second);
      return ret;
    }

  private:
    typedef nanoflann::KDTreeSingleIndexAdaptor
      <
        nanoflann::L2_Simple_Adaptor<BaseType, PointCloudAdaptor>,
        PointCloudAdaptor,
        3
      > KDTree;

    PointCloudAdaptor itsAdaptor;
    KDTree itsTree;
    float const itsEpsilon;
};
